#! /bin/sh
#  ------------------------------------------------------------
#  Company: Software- und Organisations-Service GmbH
#  Purpose: Start Script for JS7 Agent
#  ------------------------------------------------------------

set -e

### BEGIN INIT INFO
# Provides:          JS7Agent
# Required-Start:    $syslog $remote_fs
# Required-Stop:     $syslog $remote_fs
# Default-Start:     3 5
# Default-Stop:      0 1 2 6
# Description:       Start JS7 Agent
### END INIT INFO


### SETTINGS #############################################

# This variable has to point to the installation path of 
# the JS7 Agent. If this variable is not defined then the parent
# directory of this start script is used.
### NOTE: 
# This variable is required if this script is used for a  
# JS7 Agent service in /etc/init.d to locate the installation
# path of the JS7 Agent.
#
# JS7_AGENT_HOME=


# Sets the user account for the JS7 Agent.
# Without this setting the currently logged in user account is used.
# The JS7 Agent should always be started with the same user account
# to avoid problems about permissions in the JS7_AGENT_DATA folder.
# This variable has to be set if this start script is used for autostart
# to avoid that the JS7 Agent is started as the 'root' user account.
#
# JS7_AGENT_USER=


# The default port of the JS7 Agent
#
JS7_AGENT_DEFAULT_HTTP_PORT=4445


# Sets the http port for the JS7 Agent.
# Without this setting the default port 4445 is used.
# If just a port is specified then the JS7 Agent listens to all
# available network interfaces. This corresponds to 0.0.0.0:<port>.
# Use the form <ip address or hostname>:<port> to indicate
# a specific network interface the JS7 Agent should listen to.
# The command line option --http-port beats the environment
# variable JS7_AGENT_HTTP_PORT.
#
# JS7_AGENT_HTTP_PORT=


# In addition to the http port an https port for the
# JS7 Agent can be specified. If just a port is specified
# then the JS7 Agent listens to all available network interfaces.
# This corresponds to using 0.0.0.0:<port>.
# Use the form <ip address or hostname>:<port> to indicate
# a specific network interface the JS7 Agent should listen to.
# The command line option --https-port beats the environment
# variable JS7_AGENT_HTTPS_PORT.
#
# JS7_AGENT_HTTPS_PORT=


# Sets the directory used by the JS7 Agent for configuration
# files, log files, journal files and temporary files.
# This directory has to be unique for any instance of the
# JS7 Agent. The default value is
# JS7_AGENT_HOME/var_<JS7_AGENT_HTTP_PORT>.
# Check that the JS7 Agent user account has read/write
# permissions for this directory.
#
# JS7_AGENT_DATA=


# Sets the directory used by the JS7 Agent for configuration
# files. The default value is JS7_AGENT_DATA/config.
# The command line option --config-directory beats the environment
# variable JS7_AGENT_CONFIG_DIR.
# Check that the JS7 Agent user account has read/write
# permissions for this directory.
#
# JS7_AGENT_CONFIG_DIR=


# Sets the directory to which the JS7 Agent stores log files.
# The default value is JS7_AGENT_DATA/logs.
# Check that the JS7 Agent user account has read/write
# permissions for this directory.
#
# JS7_AGENT_LOGS=


# Sets the directory to which the JS7 Agent stores its PID file.
# The default value is JS7_AGENT_LOGS.
# Check that the JS7 Agent user account has read/write
# permissions for this directory.
#
# JS7_AGENT_PID_FILE_DIR=


# The working directory for the JS7 Agent is JS7_AGENT_DATA
# Sets the working directory for jobs.
# The default value is JS7_AGENT_DATA.
#
# JS7_AGENT_WORK_DIR=


# If the Agent is terminated abnormally, the still running 
# processes of the jobs are terminated with SIGTERM and after 
# the delay in seconds specified here with SIGKILL.
# The default value is 3.
#
# JS7_AGENT_SIGKILL_DELAY=
JS7_AGENT_SIGKILL_DELAY_DEFAULT=3


# Specifies the Java Runtime Environment to be used by the
# JS7 Agent. If JAVA_HOME is not set then Java is used from
# the PATH.
#
# JAVA_HOME=


# With Java 1.8 initial memory allocation has changed,
# for details see https://kb.sos-berlin.com/x/aIC9
# As a result on start-up of the JS7 Agent the amount of
# virtual memory assigned by Java depends on the physical
# memory and in most cases is by far too high.
# The environment variables JAVA_OPTIONS can be used to apply
# memory settings.
# The default value for JAVA_OPTIONS is -Xms100m -Dfile.encoding=UTF-8.
#
# JAVA_OPTIONS=

##########################################################


usage() 
{
  test -z "${JS7_AGENT_SIGKILL_DELAY}" && JS7_AGENT_SIGKILL_DELAY="${JS7_AGENT_SIGKILL_DELAY_DEFAULT}"
  echo "Usage: $(basename "$0") command [options]"
  echo "  command:"
  echo "    start            [options]"
  echo "    start-container  [options]"
  echo "    stop             [options]"
  echo "    restart          [options]"
  echo "    status           [options]"
  echo "    cancel           [options]"
  echo "    switch-over      [options]"
  echo "    cert             [cert-options]         | see https://kb.sos-berlin.com/x/jLbAAw"
  echo "  options:"
  echo "    --http-port=<[interface:]port>          | http network interface and port, default: ${JS7_AGENT_HTTP_PORT}"
  echo "    --https-port=<[interface:]port>         | https network interface and port, default: ${JS7_AGENT_HTTPS_PORT}"
  echo "    --data-directory=<directory>            | default: ${JS7_AGENT_DATA}"
  echo "    --config-directory=<directory>          | default: ${JS7_AGENT_CONFIG_DIR}"
  echo "    --sigkill-delay=<seconds>               | send SIGTERM and delayed SIGKILL signal, default: ${JS7_AGENT_SIGKILL_DELAY}"
  echo "    --timeout=<seconds>                     | timeout for terminating jobs on Agent stop"
  echo "    --java-options=<java options>           | default: ${JS7_AGENT_JAVA_OPTIONS}; see https://kb.sos-berlin.com/x/uYo7B"
  echo "  switches:"
  echo "    -f | --fail-over                        | fail-over active role on stop and restart"
  echo "    -c | --curl                             | use curl instead of the Java http client"
  echo ""
  more_info
}


more_info()
{
  echo "see https://kb.sos-berlin.com/x/ZqrAAw for more information."
}

start_agent() 
{
  JS7_AGENT_EXIT=0
  if running
  then
    echo "...JS7 Agent(${JS7_AGENT_PORT}) is already running with pid=$(get_pid)!"
    exit 0
  fi
  remove_crash_file
  remove_sigkill_delay_file
  JS7_AGENT_CLASSPATH="${JS7_AGENT_CONFIG_DIR}/patches/*:${JS7_AGENT_CONFIG_DIR}/lib/*:${JS7_AGENT_HOME}/lib/patches/*:${JS7_AGENT_HOME}/lib/user_lib/*:${JS7_AGENT_HOME}/lib/sos/*:${JS7_AGENT_HOME}/lib/3rd-party/*:${JS7_AGENT_HOME}/lib/jdbc/*"
  if [ -f "${JS7_AGENT_CONFIG_DIR}/log4j2.xml" ]
  then
    JS7_AGENT_CLASSPATH="${JS7_AGENT_CONFIG_DIR}:${JS7_AGENT_CLASSPATH}"
  else
    JS7_AGENT_CLASSPATH="${JS7_AGENT_HOME}/lib:${JS7_AGENT_CLASSPATH}"
  fi
  if [ -z "${JS7_AGENT_TZ}" ]
  then
    JS7_AGENT_TZ="$(get_timezone)"
  fi
  if [ "${JS7_AGENT_DATA}" != "$(pwd)" ]
  then
    cd "${JS7_AGENT_DATA}"
  fi
  export_env
  
  #remove_folder "${JS7_AGENT_DATA}/work"
  #create_folder "${JS7_AGENT_DATA}/work"

  "${JS7_AGENT_HOME}/bin/agent_watchdog.sh" -"${JS7_AGENT_PORT}" &
  JS7_AGENT_EXIT=$?
}

start_agent_in_container()
{
  JS7_AGENT_EXIT=0
  if running
  then
     echo "...JS7 Agent(${JS7_AGENT_PORT}) is already running with pid=$(get_pid)!"
     exit 0
  fi
  remove_crash_file
  remove_sigkill_delay_file
  JS7_AGENT_CLASSPATH="${JS7_AGENT_CONFIG_DIR}/patches/*:${JS7_AGENT_CONFIG_DIR}/lib/*:${JS7_AGENT_HOME}/lib/patches/*:${JS7_AGENT_HOME}/lib/user_lib/*:${JS7_AGENT_HOME}/lib/sos/*:${JS7_AGENT_HOME}/lib/3rd-party/*:${JS7_AGENT_HOME}/lib/jdbc/*"
  if [ -f "${JS7_AGENT_CONFIG_DIR}/log4j2.xml" ]
  then
    JS7_AGENT_CLASSPATH="${JS7_AGENT_CONFIG_DIR}:${JS7_AGENT_CLASSPATH}"
  else
    JS7_AGENT_CLASSPATH="${JS7_AGENT_HOME}/lib:${JS7_AGENT_CLASSPATH}"
  fi
 
  if [ -z "${JS7_AGENT_TZ}" ]
  then
    JS7_AGENT_TZ="$(get_timezone)"
  fi
  if [ "${JS7_AGENT_DATA}" != "$(pwd)" ]
  then
    cd "${JS7_AGENT_DATA}"
  fi
  export_env
  
  trap "shutdown_agent SIGKILL" INT TERM

  #remove_folder "${JS7_AGENT_DATA}/work"
  #create_folder "${JS7_AGENT_DATA}/work"
  
  test -z "${JS7_AGENT_KILL_SCRIPT}" || echo "--kill-script option is not longer supported"
  test -z "${JS7_AGENT_HTTPS_PORT}" || HTTPS_PORT_OPTION="--https-port=${JS7_AGENT_HTTPS_PORT}"

  echo "\"${JAVABIN}\" ${JS7_AGENT_JAVA_OPTIONS} -classpath \"${JS7_AGENT_CLASSPATH}\" js7.agent.main.AgentMain --http-port=\"${JS7_AGENT_HTTP_PORT}\" ${HTTPS_PORT_OPTION} --config-directory=\"${JS7_AGENT_CONFIG_DIR}\" --data-directory=\"${JS7_AGENT_DATA}\" --job-working-directory=\"${JS7_AGENT_WORK_DIR}\""
  "${JAVABIN}" ${JS7_AGENT_JAVA_OPTIONS} -classpath "${JS7_AGENT_CLASSPATH}" js7.agent.main.AgentMain --http-port="${JS7_AGENT_HTTP_PORT}" ${HTTPS_PORT_OPTION} --config-directory="${JS7_AGENT_CONFIG_DIR}" --data-directory="${JS7_AGENT_DATA}" --job-working-directory="${JS7_AGENT_WORK_DIR}" &
  JS7_AGENT_EXIT=$?
  
  if [ "${JS7_AGENT_EXIT}" = "0" ]
  then
    create_pid_file $!
    JS7_AGENT_PID=$!
    echo "...JS7 Agent(${JS7_AGENT_PORT}) is started with pid=${JS7_AGENT_PID}!"
    wait "${JS7_AGENT_PID}" || JS7_AGENT_EXIT=$?
    restart_container "${JS7_AGENT_PID}"
  fi
}

restart_container()
{
  echo "...JS7 Agent terminates with exit code ${JS7_AGENT_EXIT}"
  
  call_crash_pid_file_killer "$1"
  
  rm "${JS7_AGENT_PID_FILE}" >/dev/null 2>/dev/null || true
  
  #abort and restart
  if [ "${JS7_AGENT_EXIT}" = "98" ]
  then
    start_agent_in_container
  fi
  #terminate and restart
  if [ "${JS7_AGENT_EXIT}" = "97" ]
  then
    start_agent_in_container
  fi
}

stop_agent() 
{
  if with_failover
  then
  	abort_agent
  else
  	shutdown_agent ""
  fi
}

shutdown_agent()
{
  if running
  then
  	create_sigkill_delay_file
  	echo "...try to terminate the JS7 Agent(${JS7_AGENT_PORT}) with pid=${JS7_AGENT_PID}."
  	if [ "$1" = "SIGKILL" ]
    then
      #  called by trap in start-docker if container sends INT or TERM
      http_command POST "{\"TYPE\": \"ShutDown\", \"processSignal\": \"SIGKILL\"}"
    elif [ "${JS7_AGENT_STOP_TIMEOUT}" -gt "-2" ]
    then
      # waits for running jobs
      http_command POST "{\"TYPE\": \"ShutDown\"}"
    else
      http_command POST "{\"TYPE\": \"ShutDown\", \"processSignal\": \"SIGTERM\"}"
    fi
  else
    echo "...not started!"
  fi
  if [ "${JS7_AGENT_EXIT}" != "0" ]
  then
    echo "...could not terminate the JS7 Agent(${JS7_AGENT_PORT}) with pid=${JS7_AGENT_PID}!"
    exit 3
  else
    loop_counter=2
    sleep 2
    while running
    do
      if [ "${loop_counter}" = "2" ]
      then
        printf "...waiting for termination"
      fi
      if [ "${JS7_AGENT_STOP_TIMEOUT}" -gt "-1" ] && [ "${loop_counter}" -ge "${JS7_AGENT_STOP_TIMEOUT}" ]
      then
      	echo "..timeout reached"
      	echo "...try to cancel (SIGKILL) the JS7 Agent(${JS7_AGENT_PORT}) with pid=${JS7_AGENT_PID}."
      	kill_err=0
      	no_mercy "${JS7_AGENT_PID}" || kill_err=$?
      	if [ "$kill_err" = "0" ]
      	then
      	  exit 0
      	fi
      fi
      printf ".."
      loop_counter=$(( loop_counter + 2 ))
      sleep 2
    done
    echo "...terminated!"
    remove_pid_file
  fi
}

abort_agent()
{
  if running
  then
  	create_sigkill_delay_file
    echo "...try to terminate forcibly the JS7 Agent(${JS7_AGENT_PORT}) with pid=${JS7_AGENT_PID}."
    http_command POST "{\"TYPE\": \"EmergencyStop\"}" 2>/dev/null
  else
    echo "...not started!"
  fi
  sleep 2
  if running
  then
    echo "...could not abort the JS7 Agent(${JS7_AGENT_PORT}) with pid=${JS7_AGENT_PID}!"
    exit "${JS7_AGENT_EXIT}"
  else
    JS7_AGENT_EXIT=0
    echo "...forcibly terminated!"
    remove_pid_file
  fi
}

status_agent()
{
  http_command GET "?"
  #exit 3 if program is not running according LSB 3.1
  if [ "${JS7_AGENT_EXIT}" != "0" ]
  then
    echo "...JS7 Agent(${JS7_AGENT_PORT}) not started?"
    JS7_AGENT_EXIT=3
  fi
}

cancel_agent()
{
  if running
  then
  	create_sigkill_delay_file
    JS7_AGENT_PID=$(get_pid)
    #echo "...try to cancel (SIGTERM) the JS7 Agent(${JS7_AGENT_PORT}) with pid=${JS7_AGENT_PID}."
    #kill -15 "${JS7_AGENT_PID}" 2> /dev/null
    #sleep 5
    #if ps -p "${JS7_AGENT_PID}" > /dev/null 2>&1
    #then
    #  echo "...try once more with SIGKILL."
      echo "...try to cancel (SIGKILL) the JS7 Agent(${JS7_AGENT_PORT}) with pid=${JS7_AGENT_PID}."
      no_mercy "${JS7_AGENT_PID}"
    #else
    #  echo "...canceled!"
    #  remove_pid_file
    #fi
  else
    echo "...not started!" 
  fi
}

no_mercy()
{
  JS7_AGENT_STOP_TIMEOUT=-2
  kill_err=0
  kill -9 "$1" || kill_err=$?
  if [ "$kill_err" = "0" ]
  then
    echo "...canceled!"
    remove_pid_file
    exit 0
  else
    echo "...could not cancel the JS7 Agent(${JS7_AGENT_PORT}) with pid=$1!"
    exit 3
  fi
}

switch_over()
{
  if running
  then
    echo "...try to switch over the Agent cluster of JS7 Agent(${JS7_AGENT_PORT})."
    http_command POST "{\"TYPE\": \"ClusterSwitchOver\"}"
  else
    echo "...not started!"
  fi	
}

call_cert() 
{
  JS7_AGENT_LIBDIR="${JS7_AGENT_HOME}/lib"
  "${JAVABIN}" -Xmx50m -Djs7.config-directory="${JS7_AGENT_CONFIG_DIR}" -classpath "${JS7_AGENT_LIBDIR}/http-client:${JS7_AGENT_CONFIG_DIR}/patches/*:${JS7_AGENT_CONFIG_DIR}/lib/*:${JS7_AGENT_LIBDIR}/patches/*:${JS7_AGENT_LIBDIR}/sos/*:${JS7_AGENT_LIBDIR}/3rd-party/*" com.sos.cli.ExecuteRollOut "$@"
  exit $?
}

http_command()
{
  if with_curl
  then
    command_with_curl "$1" "$2"
  else
    command_with_agent_client "$2"
  fi
}

command_with_agent_client()
{
  JS7_AGENT_EXIT=0
  JS7_AGENT_LIBDIR="${JS7_AGENT_HOME}/lib"
  "${JAVABIN}" -Xmx50m -classpath "${JS7_AGENT_LIBDIR}/http-client:${JS7_AGENT_LIBDIR}/sos/*:${JS7_AGENT_LIBDIR}/3rd-party/*" js7.agent.client.main.AgentClientMain --config-directory="${JS7_AGENT_CONFIG_DIR}" --data-directory="${JS7_AGENT_DATA}" "$(get_url_cutted)" "$1" || JS7_AGENT_EXIT=$?
}

command_with_curl()
{
  JS7_AGENT_EXIT=0
  if [ "$1" = "POST" ]
  then
    curl --fail -L -X POST -d "$2" -H "X-JS7-Session: $(cat "${JS7_AGENT_DATA}/work/session-token")" -H "Content-Type: application/json" "$(get_url)/api/command" || JS7_AGENT_EXIT=$?
  else
    curl --fail -L -H "X-JS7-Session: $(cat "${JS7_AGENT_DATA}/work/session-token")" "$(get_url)/api$2" || JS7_AGENT_EXIT=$?
  fi
  echo ""
}

call_crash_pid_file_killer()
{
  JS7_AGENT_PID="$1"
  if [ "$(cat "${JS7_AGENT_DATA}/state/lock" 2>/dev/null || true)" -eq "${JS7_AGENT_PID}" ] && [ -s "${JS7_AGENT_CRASH_FILE}" ]
  then
  	test -z "${JS7_AGENT_SIGKILL_DELAY}" && JS7_AGENT_SIGKILL_DELAY="${JS7_AGENT_SIGKILL_DELAY_DEFAULT}"
  	echo "\"${JAVABIN}\" -Xmx100m -classpath \"${JS7_AGENT_CLASSPATH}\" js7.launcher.crashpidfile.CrashPidFileKiller --data-directory=\"${JS7_AGENT_DATA}\" --sigkill-delay=\"${JS7_AGENT_SIGKILL_DELAY}\""
    "${JAVABIN}" -Xmx100m -classpath "${JS7_AGENT_CLASSPATH}" js7.launcher.crashpidfile.CrashPidFileKiller --data-directory="${JS7_AGENT_DATA}" --sigkill-delay="${JS7_AGENT_SIGKILL_DELAY}" || true
  fi
}

get_pid() 
{
  if [ -f "${JS7_AGENT_PID_FILE}" ]
  then
    cat "${JS7_AGENT_PID_FILE}"
    return 0
  else
    JS7_AGENT_PID=$(get_pid_from_ps)
    echo "${JS7_AGENT_PID}"
    if [ -z "${JS7_AGENT_PID}" ]
    then
      return 1
    fi
  fi 
  return 0
}

get_pid_from_ps() 
{
  ps -ef | grep -E "js7\.agent\.main\.AgentMain.*--http-port=${JS7_AGENT_HTTP_PORT}" | grep -v "grep" | awk '{print $2}'
}

running()
{
  JS7_AGENT_PID=$(get_pid 2>/dev/null) || return 1 
  if kill -0 "${JS7_AGENT_PID}" 2>/dev/null
  then
    return 0
  else
    remove_pid_file
  fi
  return 1
}

get_timezone()
{
  if [ -f "${JS7_AGENT_HOME}/lib/classes/DefaultTimeZone.class" ]
  then
    "${JAVABIN}" -classpath "${JS7_AGENT_HOME}/lib/classes" DefaultTimeZone 2>/dev/null || echo "Etc/UTC"
  else
    echo "Etc/UTC"
  fi
}

with_failover()
{
  test -z "${WITH_FAILOVER}" && return 1
  return "${WITH_FAILOVER}"
}

with_curl()
{
  which curl >/dev/null 2>/dev/null || return 1
  return "${WITH_CURL}"
}

create_sigkill_delay_file()
{
  if [ -n "${JS7_AGENT_SIGKILL_DELAY}" ]
  then
    echo "${JS7_AGENT_SIGKILL_DELAY}" > "${JS7_AGENT_SIGKILL_DELAY_FILE}" || true
  fi
}

create_pid_file() 
{
  JS7_AGENT_PID=$1
  echo "${JS7_AGENT_PID}" > "${JS7_AGENT_PID_FILE}" || true
  if [ ! -f "${JS7_AGENT_PID_FILE}" ]
  then
    echo "...could not create pid file: \"${JS7_AGENT_PID_FILE}\""
    echo "...please check permissions for directory: \"${JS7_AGENT_PID_FILE_DIR}\"."
    more_info
    exit 3
  else
    chmod 666 "${JS7_AGENT_PID_FILE}" 2>/dev/null
  fi 
}

remove_pid_file()
{
  rm -f "${JS7_AGENT_PID_FILE}" 2>/dev/null || return 1
  return 0
}

remove_crash_file()
{
  rm -f "${JS7_AGENT_CRASH_FILE}" 2>/dev/null || return 1
  return 0
}

remove_sigkill_delay_file()
{
  rm -f "${JS7_AGENT_SIGKILL_DELAY_FILE}" 2>/dev/null || return 1
  return 0
}

mkdir_error()
{
  echo "...could not create \"$1\""
  echo "please create directory \"$1\" with read/write permissions for user account \"${JS7_AGENT_USER}\""
  MKDIR_ERROR_EXISTS=1
  more_info
}

export_env()
{
  JS7_AGENT_APPNAME=agent
  export JS7_AGENT_APPNAME
  export JS7_AGENT_USER
  export JS7_AGENT_HOME
  export JS7_AGENT_DATA
  export JS7_AGENT_CONFIG_DIR
  export JS7_AGENT_WORK_DIR
  export JS7_AGENT_PORT
  export JS7_AGENT_HTTP_PORT
  export JS7_AGENT_HTTPS_PORT
  export JS7_AGENT_CRASH_FILE
  export JS7_AGENT_KILL_SCRIPT
  export JAVABIN
  export JS7_AGENT_LOGS
  export JS7_AGENT_JAVA_OPTIONS
  export JS7_AGENT_CLASSPATH
  export JS7_AGENT_PID_FILE
  export JS7_AGENT_PID_FILE_DIR
  export JS7_AGENT_TZ
  export JS7_AGENT_SIGKILL_DELAY
  export JS7_AGENT_SIGKILL_DELAY_DEFAULT
  export JS7_AGENT_SIGKILL_DELAY_FILE
  
  JS7_YADE_HOME="${JS7_AGENT_HOME}/yade"
  JS7_YADE_BIN="${JS7_YADE_HOME}/bin/yade.sh"
  JS7_YADE_DMZ_BIN="${JS7_YADE_HOME}/bin/yade4dmz.sh"
  JS7_YADE_CONFIG_DIR="${JS7_AGENT_CONFIG_DIR}"
  export JS7_YADE_HOME
  export JS7_YADE_BIN
  export JS7_YADE_DMZ_BIN
  export JS7_YADE_CONFIG_DIR
}

get_port()
{
  echo "$1" | cut -d : -f 2 
}

get_url()
{
  if [ -f "${JS7_AGENT_DATA}/work/http-uri" ]
  then
    cat "${JS7_AGENT_DATA}/work/http-uri"
  else
    echo "http://$(add_ip_address "${JS7_AGENT_HTTP_PORT}" 127.0.0.1)/subagent"
  fi
}

get_url_cutted()
{
  echo "$(get_url | cut -d / -f 1,2,3)"
}

add_ip_address()
{
  IP_ADDRESS=$(echo "$1" | grep ":" | cut -d : -f 1)
  NEW_IP_ADDRESS="$2"
  if [ -n "$NEW_IP_ADDRESS" ]
  then
    NEW_IP_ADDRESS="${NEW_IP_ADDRESS}:"
  fi
  if [ -z "${IP_ADDRESS}" ] || [ "${IP_ADDRESS}" = "0.0.0.0" ]
  then
    echo "${NEW_IP_ADDRESS}$1"
  else
    echo "$1"
  fi
}

create_folder()
{
  if [ ! -d "$1" ]
  then
    if [ "${CURRENT_USER}" = "${JS7_AGENT_USER}" ] || [ "${CURRENT_GROUP}" = "root" ]
    then
      mkdir -p "$1" 2>/dev/null || mkdir_error "$1"
    else
      su - "${JS7_AGENT_USER}" -c "mkdir -p $1 2>/dev/null" || mkdir_error "$1"
    fi
  fi
}

remove_folder()
{
  if [ -d "$1" ]
  then
    if [ "${CURRENT_USER}" = "${JS7_AGENT_USER}" ] || [ "${CURRENT_GROUP}" = "root" ]
    then
      rm -rf "$1" 2>/dev/null || return 1
    else
      su - "${JS7_AGENT_USER}" -c "rm -rf $1 2>/dev/null" || return 1
    fi
  fi
  return 0
}

copy_file()
{
  if [ -f "$1" ] && [ ! -f "$2" ]
  then
    if [ "${CURRENT_USER}" = "${JS7_AGENT_USER}" ] || [ "${CURRENT_GROUP}" = "root" ]
    then
      cp -f "$1" "$2" 2>/dev/null || mkdir_error "$2"
    else
      su - "${JS7_AGENT_USER}" -c "cp -f $1 $2 2>/dev/null" || mkdir_error "$2"
    fi
  fi
}

is_integer()
{ 
  if [ -n "$1" ]
  then
  	test "$1" -eq "$1" 2>/dev/null
    if [ $? -ne 0 ]
    then
      echo "$1 is not an integer"
      return 1
    fi
  else
  	return 1
  fi
  return 0
}

end()
{
  if [ "${CUR_DIR}" != "$(pwd)" ]
  then
    cd "${CUR_DIR}" || true
  fi
  
  unset JS7_AGENT_USER
  unset JS7_AGENT_HOME
  unset JS7_AGENT_DATA
  unset JS7_AGENT_CONFIG_DIR
  unset JS7_AGENT_WORK_DIR
  unset JS7_AGENT_PORT
  unset JS7_AGENT_HTTP_PORT
  unset JS7_AGENT_HTTPS_PORT
  unset JS7_AGENT_CRASH_FILE
  unset JS7_AGENT_KILL_SCRIPT
  unset JAVABIN
  unset JS7_AGENT_LOGS
  unset JS7_AGENT_JAVA_OPTIONS
  unset JS7_AGENT_CLASSPATH
  unset JS7_AGENT_PID_FILE
  unset JS7_AGENT_PID_FILE_DIR
  unset JS7_AGENT_TZ
  unset JS7_YADE_HOME
  unset JS7_YADE_BIN
  unset JS7_YADE_DMZ_BIN
  unset JS7_YADE_CONFIG_DIR
  unset CUR_DIR
  unset JS7_AGENT_PID
  unset JS7_AGENT_LIBDIR
  unset JS7_AGENT_CLIENT_CLASSPATH
  unset JS7_AGENT_STOP_TIMEOUT
  unset JS7_AGENT_SIGKILL_DELAY
  unset JS7_AGENT_SIGKILL_DELAY_DEFAULT
  unset JS7_AGENT_SIGKILL_DELAY_FILE
  unset WITH_CURL
  unset MKDIR_ERROR_EXISTS
  unset IP_ADDRESS
  unset NEW_IP_ADDRESS
  unset CURRENT_USER
  unset CURRENT_GROUP
  unset JS7_AGENT_BINDIR
  unset JS7_AGENT_STARTSCRIPT
  unset HTTPS_PORT_OPTION
  
  set +e
}

trap 'end' EXIT


##########################
# set and check home
CUR_DIR="$(pwd)"
JS7_AGENT_BINDIR="$(dirname "$0")"
JS7_AGENT_BINDIR="$(cd "${JS7_AGENT_BINDIR}" >/dev/null && pwd)"
JS7_AGENT_STARTSCRIPT="${JS7_AGENT_BINDIR}/$(basename "$0")"
if [ -z "${JS7_AGENT_HOME}" ]
then
  JS7_AGENT_HOME="$(cd "${JS7_AGENT_BINDIR}/.." >/dev/null && pwd)"
fi

if [ ! -d "${JS7_AGENT_HOME}/lib" ]
then
  echo "...directory \"${JS7_AGENT_HOME}/lib\" is missing!"
  echo "did you move the start script to some other location?"
  echo "please set the JS7_AGENT_HOME environment variable in '$0'."
  more_info
  exit 3
fi


##########################
# set user
# 'whoami' on Solaris sometimes not available
CURRENT_USER=$(whoami 2>/dev/null || ( id | sed 's/.*uid=[^(]*(\([^)]*\)).*/\1/' ))
CURRENT_GROUP="$(id -g -n -r)"
if [ -z "${JS7_AGENT_USER}" ]
then
  JS7_AGENT_USER="${CURRENT_USER}"
  #if [ "${CURRENT_USER}" = "root" ]
  #then
  #  echo "The current JS7_AGENT_USER is 'root'!"
  #  echo "Please set the JS7_AGENT_USER variable in '$0'."
  #  echo "If you really want the 'root' user then set"
  #  echo "JS7_AGENT_USER=root in '$0'."
  #  more_info
  #  exit 2
  #fi
fi


##########################
# set http port etc.
JS7_AGENT_SIGNAL=""
HTTPS_PORT_OPTION=
JS7_AGENT_PORT=
WITH_CURL=1
WITH_FAILOVER=1
JS7_AGENT_STOP_TIMEOUT=-2
for param in "$@"
do
  case "$param" in
     --http-port=*)            JS7_AGENT_HTTP_PORT=$(echo "$param" | sed 's/--http-port=//' | sed 's/^"//' | sed 's/"$//')
                               ;;
     --https-port=*)           JS7_AGENT_HTTPS_PORT=$(echo "$param" | sed 's/--https-port=//' | sed 's/^"//' | sed 's/"$//')
                               ;;
     --data-directory=*)       JS7_AGENT_DATA=$(echo "$param" | sed 's/--data-directory=//' | sed 's/^"//' | sed 's/"$//' | sed 's/^\(.*\)\/$/\1/')
                               ;;
     --config-directory=*)     JS7_AGENT_CONFIG_DIR=$(echo "$param" | sed 's/--config-directory=//' | sed 's/^"//' | sed 's/"$//' | sed 's/^\(.*\)\/$/\1/')
                               ;;
     --sigkill-delay=*)        JS7_AGENT_SIGKILL_DELAY=$(echo "$param" | sed 's/--sigkill-delay=//' | sed 's/^"//' | sed 's/"$//')
                               ;;
     --timeout=*)              JS7_AGENT_STOP_TIMEOUT=$(echo "$param" | sed 's/--timeout=//' | sed 's/^"//' | sed 's/"$//')
                               ;;
     --kill-script=*)          JS7_AGENT_KILL_SCRIPT=$(echo "$param" | sed 's/--kill-script=//' | sed 's/^"//' | sed 's/"$//')
                               ;;
     --java-options=*)         JS7_AGENT_JAVA_OPTIONS=$(echo "$param" | sed 's/--java-options=//' | sed 's/^"//' | sed 's/"$//')
                               ;;
     -c | --curl)              WITH_CURL=0
                               ;;
     -f | --fail-over)         WITH_FAILOVER=0
                               ;;
     
  esac
done

JS7_AGENT_HTTP_PORT=$(echo "${JS7_AGENT_HTTP_PORT}" | sed 's/^://')
JS7_AGENT_HTTPS_PORT=$(echo "${JS7_AGENT_HTTPS_PORT}" | sed 's/^://')
test -z "${JS7_AGENT_HTTP_PORT}" && JS7_AGENT_HTTP_PORT=$(add_ip_address "${JS7_AGENT_DEFAULT_HTTP_PORT}" "")
JS7_AGENT_PORT=$(get_port "${JS7_AGENT_HTTP_PORT}")
if [ "never" = "${JS7_AGENT_STOP_TIMEOUT}" ]
then
  JS7_AGENT_STOP_TIMEOUT=-1
fi
if ! is_integer "${JS7_AGENT_STOP_TIMEOUT}"
then
  JS7_AGENT_STOP_TIMEOUT=-2
fi
if [ "${JS7_AGENT_STOP_TIMEOUT}" -lt "-1" ]
then
  JS7_AGENT_STOP_TIMEOUT=-2
fi
if [ -n "${JS7_AGENT_SIGKILL_DELAY}" ]
then
  if ! is_integer "${JS7_AGENT_SIGKILL_DELAY}" || [ "${JS7_AGENT_SIGKILL_DELAY}" -lt "0" ]
  then
    echo "--sigkill-delay has to be a positive integer. ${JS7_AGENT_SIGKILL_DELAY_DEFAULT}seconds are used"
    JS7_AGENT_SIGKILL_DELAY=
  fi
fi


##########################
# set pid file
test -z "${JS7_AGENT_DATA}" && JS7_AGENT_DATA="${JS7_AGENT_HOME}/var_${JS7_AGENT_PORT}"
test -z "${JS7_AGENT_CONFIG_DIR}" && JS7_AGENT_CONFIG_DIR="${JS7_AGENT_DATA}/config"
test -z "${JS7_AGENT_LOGS}" && JS7_AGENT_LOGS="${JS7_AGENT_DATA}/logs"
test -z "${JS7_AGENT_PID_FILE_DIR}" && JS7_AGENT_PID_FILE_DIR="${JS7_AGENT_LOGS}"
test -z "${JS7_AGENT_PID_FILE_NAME}" && JS7_AGENT_PID_FILE_NAME="agent.pid"
JS7_AGENT_PID_FILE="${JS7_AGENT_PID_FILE_DIR}/${JS7_AGENT_PID_FILE_NAME}"
MKDIR_ERROR_EXISTS=0 
create_folder "${JS7_AGENT_DATA}"
create_folder "${JS7_AGENT_LOGS}"
if [ "${JS7_AGENT_LOGS}" != "${JS7_AGENT_PID_FILE_DIR}" ]
then 
  create_folder "${JS7_AGENT_PID_FILE_DIR}"
fi
create_folder "${JS7_AGENT_DATA}/state"
create_folder "${JS7_AGENT_DATA}/work"
create_folder "${JS7_AGENT_CONFIG_DIR}/executables"
create_folder "${JS7_AGENT_CONFIG_DIR}/license"
create_folder "${JS7_AGENT_CONFIG_DIR}/lib"
create_folder "${JS7_AGENT_CONFIG_DIR}/patches"
create_folder "${JS7_AGENT_CONFIG_DIR}/private/trusted-x509-keys"
create_folder "${JS7_AGENT_CONFIG_DIR}/private/trusted-pgp-keys"
copy_file "${JS7_AGENT_HOME}/var/config/agent.conf-example" "${JS7_AGENT_CONFIG_DIR}/agent.conf"
copy_file "${JS7_AGENT_HOME}/var/config/private/private.conf-example" "${JS7_AGENT_CONFIG_DIR}/private/private.conf"
copy_file "${JS7_AGENT_HOME}/var/config/log4j2.xml-example" "${JS7_AGENT_CONFIG_DIR}/log4j2.xml-example"
# only if trusted-x509-keys folder is empty
if [ "$(ls -p "${JS7_AGENT_CONFIG_DIR}/private/trusted-x509-keys" | wc -l)" = "0" ]
then
  copy_file "${JS7_AGENT_HOME}/var/config/private/trusted-x509-keys/sos.intermediate-ca.pem" "${JS7_AGENT_CONFIG_DIR}/private/trusted-x509-keys/sos.intermediate-ca.pem"
fi
if [ "${MKDIR_ERROR_EXISTS}" = "1" ]
then
  exit 4
fi


##########################
# set crash file location
JS7_AGENT_CRASH_FILE="${JS7_AGENT_DATA}/work/crashpidfile"


##########################
# set sigkill-delay file location
JS7_AGENT_SIGKILL_DELAY_FILE="${JS7_AGENT_DATA}/work/sigkilldelay"


##########################
# set working directory
test -z "${JS7_AGENT_WORK_DIR}" && JS7_AGENT_WORK_DIR="${JS7_AGENT_DATA}"


##########################
# set java options
test -z "${JS7_AGENT_JAVA_OPTIONS}" && JS7_AGENT_JAVA_OPTIONS="${JAVA_OPTIONS}"
echo "${JS7_AGENT_JAVA_OPTIONS}" | grep -e "-Xms" >/dev/null 2>/dev/null || JS7_AGENT_JAVA_OPTIONS="${JS7_AGENT_JAVA_OPTIONS} -Xms100m"
echo "${JS7_AGENT_JAVA_OPTIONS}" | grep -e "-Dfile.encoding" >/dev/null 2>/dev/null || JS7_AGENT_JAVA_OPTIONS="${JS7_AGENT_JAVA_OPTIONS} -Dfile.encoding=UTF-8"


##########################
# set and check java
JAVABIN=$(which java 2>/dev/null || echo "")
test -n "$JAVA_HOME" && test -x "$JAVA_HOME/bin/java" && JAVABIN="$JAVA_HOME/bin/java"
if [ -z "$JAVABIN" ]
then
  echo "...could not identify Java environment"
  echo "...please set JAVA_HOME variable"
  more_info 
  exit 5
fi


##########################
# ... and action
case "$1" in
    start)
        if [ "${CURRENT_USER}" = "${JS7_AGENT_USER}" ]
        then
          start_agent
        else
          su - "${JS7_AGENT_USER}" -c "${JS7_AGENT_STARTSCRIPT} $*"
        fi
        ;;
    start-container|start_container|start-docker|start_docker)
        start_agent_in_container > "${JS7_AGENT_LOGS}/watchdog.log" 2>&1
        ;;
    stop)
        stop_agent
        ;;
    restart)
        if [ "${CURRENT_USER}" = "${JS7_AGENT_USER}" ]
        then
         stop_agent
         start_agent
        else
          su - "${JS7_AGENT_USER}" -c "${JS7_AGENT_STARTSCRIPT} $*"
        fi
        ;;
    abort)
        abort_agent
        ;;
    cancel|kill)
        cancel_agent
        ;;
    switch-over|switch_over)
        switch_over
        ;;
    status)
        shift
        status_agent "$@"
        ;;
    setup)
        echo "Directory for logs and pid files exists!"
        echo "Java version ok!"
        ;;
    cert)
        shift
        call_cert "$@"
        ;;
    *)
        usage
        ;;
esac

test -z "${JS7_AGENT_EXIT}" && JS7_AGENT_EXIT=0
exit $JS7_AGENT_EXIT