 
/*
 | Job Scheduler authentication tables for MySQL
 | SOS GmbH, 2021-11-30
*/
set SQL_MODE=ANSI_QUOTES;

/* Table for IAM_BLOCKLIST */
CREATE TABLE IF NOT EXISTS IAM_BLOCKLIST(
  "ID"                 INT  UNSIGNED NOT NULL AUTO_INCREMENT,
  "ACCOUNT_NAME"       VARCHAR(255)  NOT NULL,
  "COMMENT"            VARCHAR(255)  NULL,
  "SINCE"              DATETIME      NOT NULL,
  PRIMARY KEY ("ID")
) ENGINE=InnoDB;

/* Table for IAM_HISTORY */
CREATE TABLE IF NOT EXISTS IAM_HISTORY(
  "ID"                    INT  UNSIGNED NOT NULL AUTO_INCREMENT,
  "IDENTITY_SERVICE_ID"   INT NOT NULL DEFAULT 0,
  "ACCOUNT_NAME"          VARCHAR(255)  NOT NULL,
  "LOGIN_DATE"            DATETIME      NOT NULL,
  "LOGIN_SUCCESS"         TINYINT       NOT NULL DEFAULT 0,
  PRIMARY KEY ("ID")
) ENGINE=InnoDB;

/* Table for IAM_HISTORY_DETAILS */
CREATE TABLE IF NOT EXISTS IAM_HISTORY_DETAILS(
  "ID"                    INT  UNSIGNED NOT NULL AUTO_INCREMENT,
  "IAM_HISTORY_ID"        INT  NOT NULL,
  "IDENTITY_SERVICE_NAME" VARCHAR(255)  NOT NULL,
  "MESSAGE"               VARCHAR(2000)  NOT NULL,
  PRIMARY KEY ("ID")
) ENGINE=InnoDB;

/* Table for IAM_ACCOUNTS */
CREATE TABLE IF NOT EXISTS IAM_ACCOUNTS(
  "ID"                    INT  UNSIGNED NOT NULL AUTO_INCREMENT,
  "IDENTITY_SERVICE_ID"   INT  NOT NULL,
  "ACCOUNT_NAME"          VARCHAR(255)  NOT NULL,
  "ACCOUNT_PASSWORD"      VARCHAR(255)  NOT NULL,
  "EMAIL"                 VARCHAR(255),
  "FORCE_PASSWORD_CHANGE" TINYINT       NOT NULL DEFAULT 0,
  "DISABLED"              TINYINT       NOT NULL DEFAULT 0,
  CONSTRAINT UNIQUE_IAM_A_IN UNIQUE ("IDENTITY_SERVICE_ID","ACCOUNT_NAME"), 
  PRIMARY KEY ("ID")
) ENGINE=InnoDB;

/* Table for IAM_FIDO2_DEVICES */
CREATE TABLE IF NOT EXISTS IAM_FIDO2_DEVICES(
  "ID"                    INT  UNSIGNED NOT NULL AUTO_INCREMENT,
  "IDENTITY_SERVICE_ID"   INT     NOT NULL,
  "ACCOUNT_ID"            INT     NOT NULL,
  "PUBLIC_KEY"            VARCHAR(1024)  NOT NULL,
  "ALGORITHM"             VARCHAR(60)    NOT NULL,
  "CREDENTIAL_ID"         VARCHAR(255),
  "ORIGIN"                VARCHAR(255)   NOT NULL,
  PRIMARY KEY ("ID")
) ENGINE=InnoDB;

/* Table for IAM_ROLES */
CREATE TABLE IF NOT EXISTS IAM_ROLES(
  "ID"                   INT  UNSIGNED NOT NULL AUTO_INCREMENT,
  "IDENTITY_SERVICE_ID"  INT  NOT NULL,
  "ROLE_NAME"            VARCHAR(255)  NOT NULL,
  "ORDERING"             INT  NULL,
  CONSTRAINT UNIQUE_IAM_R_IN UNIQUE ("IDENTITY_SERVICE_ID","ROLE_NAME"), 
  PRIMARY KEY ("ID")
) ENGINE=InnoDB;

/* Table for IAM_ACCOUNT2ROLES */
CREATE TABLE IF NOT EXISTS IAM_ACCOUNT2ROLES(
  "ID"                    INT  UNSIGNED NOT NULL AUTO_INCREMENT,
  "ROLE_ID"               INT  NOT NULL,
  "ACCOUNT_ID"            INT  NOT NULL,
  PRIMARY KEY ("ID")
) ENGINE=InnoDB;

/* Table for IAM_PERMISSIONS */
CREATE TABLE IF NOT EXISTS IAM_PERMISSIONS(
  "ID"                    INT  UNSIGNED NOT NULL AUTO_INCREMENT,
  "IDENTITY_SERVICE_ID"   INT  NOT NULL,
  "CONTROLLER_ID"         VARCHAR(255) NULL,
  "ACCOUNT_ID"            INT NULL,
  "ROLE_ID"               INT NULL,
  "ACCOUNT_PERMISSION"    VARCHAR(255) NULL,
  "FOLDER_PERMISSION"     VARCHAR(255) NULL,
  "EXCLUDED"              TINYINT       NOT NULL DEFAULT 0,
  "RECURSIVE"             TINYINT       NOT NULL DEFAULT 0,

  PRIMARY KEY ("ID")
) ENGINE=InnoDB;

/* Table for IAM_IDENTITY_SERVICES */
CREATE TABLE IF NOT EXISTS IAM_IDENTITY_SERVICES(
  "ID"                     INT  UNSIGNED NOT NULL AUTO_INCREMENT,
  "IDENTITY_SERVICE_TYPE"  VARCHAR(255)  NOT NULL,
  "IDENTITY_SERVICE_NAME"  VARCHAR(255)  NOT NULL,
  "SECOND_FACTOR_IS_ID"    INT,  
  "AUTHENTICATION_SCHEME"  VARCHAR(255)  NOT NULL,
  "SECOND_FACTOR"          TINYINT       NOT NULL DEFAULT 0,
  "ORDERING"               INT  NOT NULL,
  "REQUIRED"               TINYINT       NOT NULL DEFAULT 0,
  "DISABLED"               TINYINT       NOT NULL DEFAULT 0,
  CONSTRAINT UNIQUE_IAM_S_N UNIQUE ("IDENTITY_SERVICE_NAME"), 
  PRIMARY KEY ("ID")
) ENGINE=InnoDB;
 
/* Table for IAM_FIDO2_REGISTRATIONS */
CREATE TABLE IF NOT EXISTS IAM_FIDO2_REGISTRATIONS (
  "ID" INT(10) UNSIGNED NOT NULL AUTO_INCREMENT,
  "EMAIL"               VARCHAR(255)  NOT NULL,
  "IDENTITY_SERVICE_ID" INT           NOT NULL,
  "ACCOUNT_NAME"        VARCHAR(255)  NOT NULL,
  "TOKEN"               VARCHAR(255),
  "PUBLIC_KEY"          VARCHAR(1024),
  "ALGORITHM"           VARCHAR(60),
  "CREDENTIAL_ID"       VARCHAR(255),
  "ORIGIN"              VARCHAR(255),
  "DEFERRED"            TINYINT       NOT NULL DEFAULT '0',
  "CONFIRMED"           TINYINT       NOT NULL DEFAULT '0',
  "COMPLETED"           TINYINT       NOT NULL DEFAULT '0',
  "CHALLENGE"           VARCHAR(255),
  "CREATED"             DATETIME      NOT NULL,
  CONSTRAINT UNIQUE_IAM_FIDO2_IID_AN UNIQUE ("IDENTITY_SERVICE_ID","ACCOUNT_NAME","ORIGIN"), 
  PRIMARY KEY ("ID")
) ENGINE=InnoDB;
    
 
 /* Table for IAM_FIDO2_REQUESTS */
CREATE TABLE IF NOT EXISTS IAM_FIDO2_REQUESTS (
  "ID"                  INT     UNSIGNED NOT NULL AUTO_INCREMENT,
  "IDENTITY_SERVICE_ID" INT              NOT NULL,
  "CHALLENGE"           VARCHAR(255)     NOT NULL,
  "REQUEST_ID"          VARCHAR(255)     NOT NULL,
  "CREATED"             DATETIME         NOT NULL,
  PRIMARY KEY ("ID")
) ENGINE=InnoDB; 
      
  
 
 
 
/*
 | Job Scheduler daily_plan tables for MySQL
 | SOS GmbH, 2020-05-06
*/
set SQL_MODE=ANSI_QUOTES;

/* Table for DPL_ORDERS */
CREATE TABLE IF NOT EXISTS DPL_ORDERS(
  "ID"                      INT  UNSIGNED NOT NULL AUTO_INCREMENT,
  "SUBMISSION_HISTORY_ID"   INT           NOT NULL,
  "CONTROLLER_ID"           VARCHAR(100)  NOT NULL,
  "WORKFLOW_NAME"           VARCHAR(255)  DEFAULT '.' NOT NULL,
  "WORKFLOW_PATH"           VARCHAR(255)  DEFAULT '.' NOT NULL,
  "WORKFLOW_FOLDER"         VARCHAR(255)  DEFAULT '.' NOT NULL,
  "ORDER_ID"                VARCHAR(255)  DEFAULT '.' NOT NULL,
  "ORDER_NAME"              VARCHAR(30)   DEFAULT '.' NOT NULL,
  "SCHEDULE_NAME"           VARCHAR(255)  DEFAULT '.' NOT NULL,
  "SCHEDULE_PATH"           VARCHAR(255)  DEFAULT '.' NOT NULL,
  "SCHEDULE_FOLDER"         VARCHAR(255)  DEFAULT '.' NOT NULL,
  "CALENDAR_ID"             INT           NOT NULL,
  "START_MODE"              TINYINT       NOT NULL DEFAULT 0,
  "SUBMITTED"               TINYINT       NOT NULL DEFAULT 0,
  "SUBMIT_TIME"             DATETIME      NULL,
  "PERIOD_BEGIN"            DATETIME      NULL,
  "PERIOD_END"              DATETIME      NULL,
  "REPEAT_INTERVAL"         INT           NULL,
  "PLANNED_START"           DATETIME      NOT NULL,
  "EXPECTED_END"            DATETIME      NULL,
  "ORDER_PARAMETERISATION"  VARCHAR(1000) NULL,
  "CREATED"                 DATETIME      NOT NULL,
  "MODIFIED"                DATETIME      NOT NULL,
  INDEX IDX_DPL_O_SHID("SUBMISSION_HISTORY_ID"),
  INDEX IDX_DPL_O_ON("ORDER_NAME"),
  INDEX IDX_DPL_O_OID("ORDER_ID"),
  INDEX IDX_DPL_O_PSCID("PLANNED_START","CONTROLLER_ID"),
  INDEX IDX_DPL_O_WN("WORKFLOW_NAME"),  
  PRIMARY KEY ("ID")
) ENGINE=InnoDB;

/* Table for DPL_SUBMISSIONS */
CREATE TABLE IF NOT EXISTS DPL_SUBMISSIONS(
  "ID"                    INT  UNSIGNED NOT NULL AUTO_INCREMENT,
  "CONTROLLER_ID"         VARCHAR(100)  NOT NULL,
  "SUBMISSION_FOR_DATE"   DATETIME      NOT NULL,
  "USER_ACCOUNT"          VARCHAR(255)  NOT NULL,
  "CREATED"               DATETIME      NOT NULL,
  INDEX IDX_DPL_S_SFDCID("SUBMISSION_FOR_DATE","CONTROLLER_ID"),  
  PRIMARY KEY ("ID")
) ENGINE=InnoDB;

/* Table for DPL_HISTORY */
CREATE TABLE IF NOT EXISTS DPL_HISTORY(
  "ID"                    INT  UNSIGNED NOT NULL AUTO_INCREMENT,
  "CONTROLLER_ID"         VARCHAR(100)  NOT NULL,
  "SUBMITTED"             TINYINT       NOT NULL DEFAULT 0,
  "MESSAGE"               VARCHAR(2000) NULL,
  "DAILY_PLAN_DATE"       DATETIME      NOT NULL,
  "ORDER_ID"              VARCHAR(255)  NULL,
  "WORKFLOW_PATH"         VARCHAR(255)  NULL,
  "WORKFLOW_FOLDER"       VARCHAR(255)  NULL,
  "SCHEDULED_FOR"         DATETIME      NULL, 
  "USER_ACCOUNT"          VARCHAR(255)  NOT NULL,
  "SUBMISSION_TIME"       DATETIME      NOT NULL,
  "CREATED"               DATETIME      NOT NULL,
  INDEX IDX_DPL_H_DPDSCID("DAILY_PLAN_DATE","SUBMITTED","CONTROLLER_ID"),
  INDEX IDX_DPL_H_OID("ORDER_ID"),
  PRIMARY KEY ("ID")
) ENGINE=InnoDB;

/* Table for DPL_ORDER_VARIABLES */
CREATE TABLE IF NOT EXISTS DPL_ORDER_VARIABLES(
  "ID"                    INT  UNSIGNED NOT NULL AUTO_INCREMENT,
  "CONTROLLER_ID"         VARCHAR(100)  NOT NULL,
  "ORDER_ID"              VARCHAR(255)  NOT NULL,
  "VARIABLE_VALUE"        LONGTEXT      NOT NULL,
  "CREATED"               DATETIME      NOT NULL,
  "MODIFIED"              DATETIME      NOT NULL,
  INDEX IDX_DPL_OV_OIDCID("ORDER_ID","CONTROLLER_ID"),
  PRIMARY KEY ("ID")
) ENGINE=InnoDB;

/* Table for DPL_PROJECTIONS */
CREATE TABLE IF NOT EXISTS DPL_PROJECTIONS(
  "ID"                    MEDIUMINT(6) UNSIGNED NOT NULL,
  "CONTENT"               LONGBLOB              NOT NULL,
  "CREATED"               DATETIME              NOT NULL,
  PRIMARY KEY ("ID")
) ENGINE=InnoDB;
 
/*
 | Job Scheduler Deployment tables for MySQL
 | SOS GmbH, 2019-10-04
*/
set SQL_MODE=ANSI_QUOTES;

CREATE TABLE IF NOT EXISTS DEP_HISTORY (
    "ID"              INT     UNSIGNED    NOT NULL AUTO_INCREMENT,
    "ACCOUNT"         VARCHAR(100)        NOT NULL,
    "PATH"            VARCHAR(255)        NOT NULL,
    "FOLDER"          VARCHAR(255)        NOT NULL,
    "NAME"            VARCHAR(255)        NOT NULL DEFAULT '',
    "TITLE"           VARCHAR(255)        NULL,
    "TYPE"            TINYINT             NOT NULL,
    "INV_CID"         INT     UNSIGNED    NOT NULL,
    "INV_IID"         INT     UNSIGNED    NOT NULL,   /* Inventory Instance ID */
    "CONTROLLER_ID"   VARCHAR(100)        NOT NULL,
    "CONTENT"         LONGTEXT            NOT NULL,
    "INV_CONTENT"     LONGTEXT            NOT NULL DEFAULT '',
    "SIGNATURE"       VARCHAR(2000)       NOT NULL,
    "COMMIT_ID"       VARCHAR(255)        NULL,
    "VERSION"         VARCHAR(50)         NULL,
    "OPERATION"       TINYINT             NULL,       /* UPDATE, DELETE */
    "STATE"           TINYINT             NOT NULL,   /* DEPLOYED, NOT_DEPLOYED */
    "ERROR_MESSAGE"   VARCHAR(255)        NULL,       /* ERROR MESSAGE */
    "DEPLOYMENT_DATE" DATETIME            NOT NULL,
    "DELETED_DATE"    DATETIME            NULL,
    "AUDITLOG_ID"     INT     UNSIGNED    NULL,
    INDEX IDX_DH_P ("PATH"),
    INDEX IDX_DH_N ("NAME"),
    INDEX IDX_DH_F ("FOLDER"),
    INDEX IDX_DH_INVCID ("INV_CID"),
    CONSTRAINT UNIQUE_DH_NTCC UNIQUE ("NAME", "TYPE", "CONTROLLER_ID", "COMMIT_ID"),
    PRIMARY KEY ("ID")
) ENGINE=InnoDB;

CREATE TABLE IF NOT EXISTS DEP_KEYS(
    "ID"                    INT     UNSIGNED NOT NULL AUTO_INCREMENT,
    "KEY_TYPE"              TINYINT          NOT NULL, /* PRIVATE, PUBLIC, X.509 */ 
    "KEY_ALG"               TINYINT          NOT NULL, /* PGP, RSA, ECDSA */
    "KEY"                   VARCHAR(4000)    NULL,
    "CERTIFICATE"           VARCHAR(4000)    NULL,     /* X.509 Certificate*/
    "ACCOUNT"               VARCHAR(255)     NOT NULL,
    "SECLVL"                TINYINT          NOT NULL DEFAULT 0,
    INDEX IDX_DK_A ("ACCOUNT"),
    INDEX IDX_DK_SL ("SECLVL"),
    CONSTRAINT DEP_K_UNIQUE UNIQUE ("ACCOUNT", "KEY_TYPE", "SECLVL"),
    PRIMARY KEY ("ID")
) ENGINE=InnoDB;

/* Table for Signatures */
CREATE TABLE IF NOT EXISTS DEP_SIGNATURES(
    "ID"                    INT      UNSIGNED NOT NULL AUTO_INCREMENT,
    "INV_CID"               INT      UNSIGNED     NULL, 
    "DEP_HID"               INT      UNSIGNED     NULL,
    "SIGNATURE"             VARCHAR(2000)     NOT NULL,
    "ACCOUNT"               VARCHAR(255)      NOT NULL,
    "MODIFIED"              DATETIME          NOT NULL,
    INDEX IDX_DS_A ("ACCOUNT"),
    PRIMARY KEY ("ID")
) ENGINE=InnoDB;

/* Table for Versions */
CREATE TABLE IF NOT EXISTS DEP_VERSIONS(
    "ID"                    INT      UNSIGNED NOT NULL AUTO_INCREMENT,
    "INV_CID"               INT      UNSIGNED     NULL, 
    "DEP_HID"               INT      UNSIGNED     NULL,
    "VERSION"               VARCHAR(100)      NOT NULL,
    "MODIFIED"              DATETIME          NOT NULL,
    INDEX IDX_DV_V ("VERSION"),
    PRIMARY KEY ("ID")
) ENGINE=InnoDB;

/* Table for commitIds to set for SecLvl HIGH before export */
CREATE TABLE IF NOT EXISTS DEP_COMMIT_IDS(
    "ID"                    INT      UNSIGNED  NOT NULL AUTO_INCREMENT,
    "INV_CID"               INT      UNSIGNED      NULL, 
    "DEP_HID"               INT      UNSIGNED      NULL,
    "CFG_PATH"              VARCHAR(255)       NOT NULL,
    "COMMIT_ID"             VARCHAR(255)       NOT NULL,
    PRIMARY KEY ("ID")
) ENGINE=InnoDB;

/* Table for uncommited changes due to error or unavailability of the specified controller */
CREATE TABLE IF NOT EXISTS DEP_SUBMISSIONS(
    "ID"              INT      UNSIGNED   NOT NULL AUTO_INCREMENT,
    "DEP_HID"         INT      UNSIGNED   NOT NULL,
    "ACCOUNT"         VARCHAR(100)        NOT NULL,
    "FOLDER"          VARCHAR(255)        NOT NULL,
    "PATH"            VARCHAR(255)        NOT NULL,
    "TYPE"            TINYINT             NOT NULL,
    "INV_CID"         INT      UNSIGNED   NOT NULL,
    "INV_IID"         INT      UNSIGNED   NOT NULL,   /* Inventory Instance ID */
    "CONTROLLER_ID"   VARCHAR(100)        NOT NULL,
    "CONTENT"         LONGTEXT            NOT NULL,
    "SIGNATURE"       VARCHAR(2000)       NOT NULL,
    "COMMIT_ID"       VARCHAR(255)        NULL,
    "VERSION"         VARCHAR(50)         NULL,
    "OPERATION"       TINYINT             NULL,       /* UPDATE, DELETE */
    "CREATED"         DATETIME            NOT NULL,
    "DELETED_DATE"    DATETIME            NULL,
    CONSTRAINT DEP_SUB_UNIQUE UNIQUE ("PATH", "CONTROLLER_ID", "COMMIT_ID", "CREATED"),
    PRIMARY KEY ("ID")
) ENGINE=InnoDB;
/*
 | Job Scheduler encipherment tables for MySQL
 | SOS GmbH, 2024-05-22
*/
set SQL_MODE=ANSI_QUOTES;

CREATE TABLE IF NOT EXISTS ENC_CERTIFICATES (
  "ALIAS"                   VARCHAR(255)  NOT NULL,
  "CERT"                    VARCHAR(4000) NOT NULL,
  "PK_PATH"                 VARCHAR(255)  NULL,
  PRIMARY KEY ("ALIAS")
) ENGINE=InnoDB;

CREATE TABLE IF NOT EXISTS ENC_AGENT_CERTIFICATES (
  "AGENT_ID"                VARCHAR(255)  NOT NULL,
  "CERT_ALIAS"              VARCHAR(255)  NOT NULL,  /* PK of ENC_CERTIFICATES */
  PRIMARY KEY ("AGENT_ID", "CERT_ALIAS")
) ENGINE=InnoDB;


/*
 | JobScheduler History Interface tables for MySQL 5.0
*/
set SQL_MODE=ANSI_QUOTES;

/* Table for HISTORY_CONTROLLERS */
CREATE TABLE IF NOT EXISTS HISTORY_CONTROLLERS(
    "READY_EVENT_ID"            BIGINT(16)      UNSIGNED    NOT NULL,
    "CONTROLLER_ID"             VARCHAR(100)                NOT NULL,
    "URI"                       VARCHAR(255)                NOT NULL,
    "TIMEZONE"                  VARCHAR(100)                NOT NULL,
    "TOTAL_RUNNING_TIME"        BIGINT(16)      UNSIGNED    NOT NULL,   /* in milliseconds */
    "READY_TIME"                DATETIME                    NOT NULL,
    "SHUTDOWN_TIME"             DATETIME,
    "LAST_KNOWN_TIME"           DATETIME,
    "CREATED"                   DATETIME                    NOT NULL,
    PRIMARY KEY ("READY_EVENT_ID","CONTROLLER_ID")
) ENGINE=InnoDB;

/* Table for HISTORY_AGENTS */
CREATE TABLE IF NOT EXISTS HISTORY_AGENTS(
    "READY_EVENT_ID"            BIGINT(16)      UNSIGNED    NOT NULL,
    "CONTROLLER_ID"             VARCHAR(100)                NOT NULL, 
    "AGENT_ID"                  VARCHAR(255)                NOT NULL,
    "URI"                       VARCHAR(255)                NOT NULL,
    "TIMEZONE"                  VARCHAR(100)                NOT NULL,
    "READY_TIME"                DATETIME                    NOT NULL,
    "COUPLING_FAILED_TIME"      DATETIME,
    "COUPLING_FAILED_MESSAGE"   VARCHAR(500),
    "SHUTDOWN_TIME"             DATETIME,
    "LAST_KNOWN_TIME"           DATETIME,
    "CREATED"                   DATETIME                    NOT NULL,
    INDEX IDX_HA_AID("AGENT_ID"),
    PRIMARY KEY ("READY_EVENT_ID","CONTROLLER_ID")
) ENGINE=InnoDB;

/* Table for HISTORY_ORDERS */
CREATE TABLE IF NOT EXISTS HISTORY_ORDERS(
    "ID"                        BIGINT(20)      UNSIGNED    NOT NULL    AUTO_INCREMENT,
    "CONTROLLER_ID"             VARCHAR(100)                NOT NULL,
    "ORDER_ID"                  VARCHAR(255)                NOT NULL,
    "WORKFLOW_PATH"             VARCHAR(255)                NOT NULL,
    "WORKFLOW_VERSION_ID"       VARCHAR(255)                NOT NULL,   /* #2019-06-13T08:43:29Z */
    "WORKFLOW_POSITION"         VARCHAR(255)                NOT NULL,   /* 1#fork_1#0 */
    "WORKFLOW_FOLDER"           VARCHAR(255)                NOT NULL,
    "WORKFLOW_NAME"             VARCHAR(255)                NOT NULL,
    "WORKFLOW_TITLE"            VARCHAR(255),                           
    "MAIN_PARENT_ID"            BIGINT(20)      UNSIGNED    NOT NULL,   /* HISTORY_ORDERS.ID of the main order */
    "PARENT_ID"                 BIGINT(20)      UNSIGNED    NOT NULL,   /* HISTORY_ORDERS.ID of the parent order */
    "PARENT_ORDER_ID"           VARCHAR(255)                NOT NULL,   /* HISTORY_ORDERS.ORDER_ID */
    "HAS_CHILDREN"              TINYINT         UNSIGNED    NOT NULL,
    "RETRY_COUNTER"             INT             UNSIGNED    NOT NULL,
    "NAME"                      VARCHAR(255)                NOT NULL,   /* orderId or branchId by fork */
    "START_CAUSE"               VARCHAR(50)                 NOT NULL,   /* order, fork, file_trigger, setback, unskip, unstop */
    "START_TIME_SCHEDULED"      DATETIME,                               
    "START_TIME"                DATETIME                    NOT NULL,
    "START_WORKFLOW_POSITION"   VARCHAR(255)                NOT NULL,
    "START_EVENT_ID"            BIGINT(16)      UNSIGNED    NOT NULL,   /* OrderStarted eventId */
    "START_VARIABLES"           LONGTEXT,                               /*  */
    "CURRENT_HOS_ID"            BIGINT(20)      UNSIGNED    NOT NULL,   /* HISTORY_ORDER_STEPS.ID */
    "END_TIME"                  DATETIME,
    "END_WORKFLOW_POSITION"     VARCHAR(255),
    "END_EVENT_ID"              BIGINT(16)      UNSIGNED,               /* OrderFinisched eventId */
    "END_HOS_ID"                BIGINT(20)      UNSIGNED    NOT NULL,   /* HISTORY_ORDER_STEPS.ID */
    "END_RETURN_CODE"           INT(10),
    "END_MESSAGE"               VARCHAR(500),                           
    "SEVERITY"                  TINYINT(1)      UNSIGNED    NOT NULL,   /* 0,1,2 */
    "STATE"                     TINYINT         UNSIGNED    NOT NULL,   /* */
    "STATE_TIME"                DATETIME                    NOT NULL,
    "STATE_TEXT"                VARCHAR(255),                           
    "HAS_STATES"                TINYINT         UNSIGNED    NOT NULL,   /* has entries in HISTORY_ORDER_STATES */
    "ERROR"                     TINYINT                     NOT NULL,
    "ERROR_STATE"               VARCHAR(20),                            /* failed, disrupted ... - event outcome*/
    "ERROR_REASON"              VARCHAR(50),                            /* other ... - event outcome*/
    "ERROR_RETURN_CODE"         INT(10),
    "ERROR_CODE"                VARCHAR(50),                            
    "ERROR_TEXT"                VARCHAR(500),                           
    "LOG_ID"                    BIGINT(20)      UNSIGNED    NOT NULL,   /* HISTORY_LOGS.ID */
    "CONSTRAINT_HASH"           CHAR(64)                    NOT NULL,   /* hash from "CONTROLLER_ID", "ORDER_ID"*/
    "CREATED"                   DATETIME                    NOT NULL,
    "MODIFIED"                  DATETIME                    NOT NULL,
    CONSTRAINT UNIQUE_HO_CH UNIQUE ("CONSTRAINT_HASH"), /* used by history*/
    INDEX IDX_HO_OID("ORDER_ID"),
    INDEX IDX_HO_WPATH("WORKFLOW_PATH"),
    INDEX IDX_HO_WNAME("WORKFLOW_NAME"),
    INDEX IDX_HO_STIME_PID_SY_CID("START_TIME","PARENT_ID","SEVERITY","CONTROLLER_ID"),
    INDEX IDX_HO_LID("LOG_ID"),
    PRIMARY KEY ("ID")
) ENGINE=InnoDB;

/* Table for HISTORY_ORDER_STATES */
CREATE TABLE IF NOT EXISTS HISTORY_ORDER_STATES(
    "ID"                        BIGINT(20)      UNSIGNED    NOT NULL    AUTO_INCREMENT,
    "HO_MAIN_PARENT_ID"         BIGINT(20)      UNSIGNED    NOT NULL,   /* HISTORY_ORDERS.MAIN_PARENT_ID */
    "HO_PARENT_ID"              BIGINT(20)      UNSIGNED    NOT NULL,   /* HISTORY_ORDERS.PARENT_ID */
    "HO_ID"                     BIGINT(20)      UNSIGNED    NOT NULL,   /* HISTORY_ORDERS.ID */
    "STATE"                     TINYINT         UNSIGNED    NOT NULL,   
    "STATE_TIME"                DATETIME                    NOT NULL,
    "STATE_EVENT_ID"            CHAR(16)                    NOT NULL,   
    "STATE_CODE"                VARCHAR(50),                           
    "STATE_TEXT"                VARCHAR(255),                           
    "CREATED"                   DATETIME                    NOT NULL,
    INDEX IDX_HOSTATES_HOID("HO_ID"),
    PRIMARY KEY ("ID")
) ENGINE=InnoDB;

/* Table for HISTORY_ORDER_STEPS */
CREATE TABLE IF NOT EXISTS HISTORY_ORDER_STEPS(
    "ID"                        BIGINT(20)      UNSIGNED    NOT NULL    AUTO_INCREMENT,
    "CONTROLLER_ID"             VARCHAR(100)                NOT NULL,
    "ORDER_ID"                  VARCHAR(255)                NOT NULL,
    "WORKFLOW_PATH"             VARCHAR(255)                NOT NULL,
    "WORKFLOW_VERSION_ID"       VARCHAR(255)                NOT NULL,   /* #2019-06-13T08:43:29Z */
    "WORKFLOW_POSITION"         VARCHAR(255)                NOT NULL,   /* 1#fork_1#3 */
    "WORKFLOW_FOLDER"           VARCHAR(255)                NOT NULL,
    "WORKFLOW_NAME"             VARCHAR(255)                NOT NULL,
    "HO_MAIN_PARENT_ID"         BIGINT(20)      UNSIGNED    NOT NULL,   /* HISTORY_ORDERS.MAIN_PARENT_ID */
    "HO_ID"                     BIGINT(20)      UNSIGNED    NOT NULL,   /* HISTORY_ORDERS.ID */
    "POSITION"                  INT             UNSIGNED    NOT NULL,   /* 3 - last position from WORKFLOW_POSITION */
    "RETRY_COUNTER"             INT             UNSIGNED    NOT NULL,
    "JOB_NAME"                  VARCHAR(255)                NOT NULL,
    "JOB_LABEL"                 VARCHAR(255)                NOT NULL,
    "JOB_TITLE"                 VARCHAR(255),                           
    "JOB_NOTIFICATION"          VARCHAR(1000),
    "CRITICALITY"               TINYINT         UNSIGNED    NOT NULL,
    "AGENT_ID"                  VARCHAR(255)                NOT NULL,
    "AGENT_NAME"                VARCHAR(255),
    "AGENT_URI"                 VARCHAR(255)                NOT NULL,
    "SUBAGENT_CLUSTER_ID"       VARCHAR(255),
    "START_CAUSE"               VARCHAR(50)                 NOT NULL,   /* order, file_trigger, setback, unskip, unstop */
    "START_TIME"                DATETIME                    NOT NULL,
    "START_EVENT_ID"            BIGINT(16)      UNSIGNED    NOT NULL,   /* ProcessingStarted eventId */
    "START_VARIABLES"           LONGTEXT,                          
    "END_TIME"                  DATETIME,
    "END_EVENT_ID"              BIGINT(16)      UNSIGNED,               /* Processed eventId */
    "END_VARIABLES"             LONGTEXT,
    "RETURN_CODE"               INT(10),
    "SEVERITY"                  TINYINT(1)      UNSIGNED    NOT NULL,   /* 0,1,2 */
    "ERROR"                     TINYINT                     NOT NULL,   
    "ERROR_STATE"               VARCHAR(20),                            /* failed, disrupted ... - event outcome*/
    "ERROR_REASON"              VARCHAR(50),                            /* other ... - event outcome*/
    "ERROR_CODE"                VARCHAR(50),                            
    "ERROR_TEXT"                VARCHAR(500),                           
    "LOG_ID"                    BIGINT(20)      UNSIGNED    NOT NULL,   /* HISTORY_LOGS.ID */
    "CONSTRAINT_HASH"           CHAR(64)                    NOT NULL,   /* hash from "CONTROLLER_ID", "ORDER_ID", "WORKFLOW_POSITION"*/
    "CREATED"                   DATETIME                    NOT NULL,
    "MODIFIED"                  DATETIME                    NOT NULL,
    CONSTRAINT UNIQUE_HOSTEPS_CH UNIQUE ("CONSTRAINT_HASH"),    /* used by history*/
    INDEX IDX_HOSTEPS_STIME_SY_CID("START_TIME","SEVERITY","CONTROLLER_ID"),
    INDEX IDX_HOSTEPS_HOID("HO_ID"),
    INDEX IDX_HOSTEPS_HOMPID_WPOS("HO_MAIN_PARENT_ID","WORKFLOW_POSITION"),
    INDEX IDX_HOSTEPS_WPATH("WORKFLOW_PATH"),
    INDEX IDX_HOSTEPS_WNAME("WORKFLOW_NAME"),
    INDEX IDX_HOSTEPS_CY_JN("CRITICALITY","JOB_NAME"),
    PRIMARY KEY ("ID")
) ENGINE=InnoDB;

/* Table for HISTORY_LOGS */
CREATE TABLE IF NOT EXISTS HISTORY_LOGS(
    "ID"                        BIGINT(20)      UNSIGNED    NOT NULL    AUTO_INCREMENT,
    "CONTROLLER_ID"             VARCHAR(100)                NOT NULL,
    "HO_MAIN_PARENT_ID"         BIGINT(20)      UNSIGNED    NOT NULL,  /* HISTORY_ORDERS.MAIN_PARENT_ID */
    "HO_ID"                     BIGINT(20)      UNSIGNED    NOT NULL,  /* HISTORY_ORDERS.ID */
    "HOS_ID"                    BIGINT(20)      UNSIGNED    NOT NULL,  /* HISTORY_ORDER_STEPS.ID */
    "COMPRESSED"                TINYINT                     NOT NULL,
    "FILE_BASENAME"             VARCHAR(255)                NOT NULL,
    "FILE_SIZE_UNCOMPRESSED"    INT             UNSIGNED    NOT NULL,
    "FILE_LINES_UNCOMPRESSED"   INT             UNSIGNED    NOT NULL,
    "FILE_CONTENT"              LONGBLOB                    NOT NULL,
    "CREATED"                   DATETIME                    NOT NULL,
    INDEX IDX_HLOGS_HOID("HO_ID"),
    INDEX IDX_HLOGS_HOMPID("HO_MAIN_PARENT_ID"),
    PRIMARY KEY ("ID")
) ENGINE=InnoDB;

/* Table for HISTORY_ORDER_TAGS */
CREATE TABLE IF NOT EXISTS HISTORY_ORDER_TAGS (
    "ID"                        BIGINT(20)      UNSIGNED    NOT NULL    AUTO_INCREMENT,
    "HO_ID"                     BIGINT(20)      UNSIGNED    NOT NULL,  /* HISTORY_ORDERS.ID */
    "CONTROLLER_ID"             VARCHAR(100)                NOT NULL,
    "ORDER_ID"                  VARCHAR(255)                NOT NULL,
    "TAG_NAME"                  VARCHAR(255)                NOT NULL,
    "ORDERING"                  INT                         NOT NULL,
    "GROUP_ID"                  INT             UNSIGNED    NOT NULL DEFAULT 0,
    "START_TIME"                DATETIME                    NOT NULL,
    PRIMARY KEY ("ID")
) ENGINE=InnoDB;
set SQL_MODE=ANSI_QUOTES;

/* HISTORY_ORDER_STEPS columns */
START TRANSACTION; 
set @exist := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'HISTORY_ORDER_STEPS' and column_name = 'START_VARIABLES' and upper(column_type) = 'LONGTEXT' and table_schema = database());
set @sqlstatement := if( @exist < 1, 'ALTER TABLE HISTORY_ORDER_STEPS MODIFY COLUMN "START_VARIABLES" LONGTEXT NULL;', 'select ''INFO: START_VARIABLES LONGTEXT found''');
PREPARE statement FROM @sqlstatement;
EXECUTE statement;
COMMIT;
/*
 | Job Scheduler Inventory tables for MySQL
 | SOS GmbH, 2019-10-04
*/
set SQL_MODE=ANSI_QUOTES;

CREATE TABLE IF NOT EXISTS INV_OPERATING_SYSTEMS (
  "ID"                      INT  UNSIGNED NOT NULL AUTO_INCREMENT,
  "HOSTNAME"                VARCHAR(100)  NOT NULL,
  "NAME"                    VARCHAR(50)   NULL,
  "ARCHITECTURE"            VARCHAR(255)  NULL,
  "DISTRIBUTION"            VARCHAR(255)  NULL,
  "MODIFIED"                DATETIME      NOT NULL,
  CONSTRAINT UNIQUE_IOS_HOST UNIQUE ("HOSTNAME"),
  PRIMARY KEY ("ID")
) ENGINE=InnoDB;

/* Table INV_JS_INSTANCES */
CREATE TABLE IF NOT EXISTS INV_JS_INSTANCES (
    "ID"                    INT  UNSIGNED NOT NULL AUTO_INCREMENT,
    "CONTROLLER_ID"         VARCHAR(100)  NOT NULL,
    "SECURITY_LEVEL"        TINYINT       NOT NULL DEFAULT 0, /* 0=LOW, 1=MEDIUM, 2=HIGH */
    "URI"                   VARCHAR(255)  NOT NULL,
    "CLUSTER_URI"           VARCHAR(255)  NULL,
    "OS_ID"                 INT  UNSIGNED NOT NULL,
    "VERSION"               VARCHAR(30)   NULL,
    "JAVA_VERSION"          VARCHAR(30)   NULL,
    "STARTED_AT"            DATETIME      NULL,
    "TITLE"                 VARCHAR(30)   NULL,
    "IS_CLUSTER"            TINYINT       NOT NULL DEFAULT 0, /* 0=standalone, 1=cluster */
    "IS_PRIMARY"            TINYINT       NOT NULL DEFAULT 1, /* 1=primary, 0=backup */
    "CERTIFICATE"           VARCHAR(4000) NULL,     		  /* X.509 Certificate*/
    "MODIFIED"              DATETIME      NOT NULL,
    INDEX IDX_IJI_CID("CONTROLLER_ID"),
    CONSTRAINT UNIQUE_IJI_CCP UNIQUE ("CONTROLLER_ID", "IS_CLUSTER", "IS_PRIMARY"),
    PRIMARY KEY ("ID")
) ENGINE=InnoDB;

/* Table INV_AGENT_INSTANCES */
CREATE TABLE IF NOT EXISTS INV_AGENT_INSTANCES (
    "ID"                    INT  UNSIGNED NOT NULL AUTO_INCREMENT,
    "AGENT_ID"              VARCHAR(255)  NOT NULL, /* (technical) agentName for the Controller in Job's configuration. Cannot overwritten*/
    "AGENT_NAME"            VARCHAR(255)  NOT NULL, /* (displayed) agentName for JOC in Job's configuration */
    "URI"                   VARCHAR(255)  NOT NULL,
    "PROCESS_LIMIT"         INT           NULL,
    "ORDERING"              INT           NOT NULL DEFAULT 0,
    "HIDDEN"                TINYINT       NOT NULL DEFAULT 0, /* boolean */
    "DISABLED"              TINYINT       NOT NULL DEFAULT 0, /* boolean */
    "DEPLOYED"              TINYINT       NOT NULL DEFAULT 0, /* boolean */
    "IS_WATCHER"            TINYINT       NOT NULL DEFAULT 0,
    "CONTROLLER_ID"         VARCHAR(100)  NOT NULL,
    "OS_ID"                 INT  UNSIGNED NOT NULL,
    "TITLE"                 VARCHAR(255)  NULL,
    "VERSION"               VARCHAR(30)   NULL,
    "JAVA_VERSION"          VARCHAR(30)   NULL,
    "CERTIFICATE"           VARCHAR(4000) NULL,     /* X.509 Certificate*/
    "STARTED_AT"            DATETIME      NULL,
    "MODIFIED"              DATETIME      NOT NULL,
    CONSTRAINT UNIQUE_IAI_A UNIQUE ("AGENT_ID"),
    PRIMARY KEY ("ID")
) ENGINE=InnoDB;

/* Table INV_AGENT_NAME_ALIASES */
CREATE TABLE IF NOT EXISTS INV_AGENT_NAME_ALIASES (
    "AGENT_ID"              VARCHAR(255)  NOT NULL, /* (technical) agentName for the Controller in Job's configuration. Cannot overwritten*/
    "AGENT_NAME"            VARCHAR(255)  NOT NULL, /* (displayed) agentName for JOC in Job's configuration */
    PRIMARY KEY ("AGENT_ID", "AGENT_NAME")
) ENGINE=InnoDB;

/* Table INV_SUBAGENT_INSTANCES */
CREATE TABLE IF NOT EXISTS INV_SUBAGENT_INSTANCES (
    "ID"                    INT UNSIGNED  NOT NULL AUTO_INCREMENT,
    "SUB_AGENT_ID"          VARCHAR(255)  NOT NULL,
    "AGENT_ID"              VARCHAR(255)  NOT NULL,
    "URI"                   VARCHAR(255)  NOT NULL,
    "IS_DIRECTOR"           TINYINT       NOT NULL DEFAULT 0, /* 0=No director, 1=primary director, 2=standby director */
    "ORDERING"              INT           NOT NULL DEFAULT 0,
    "DISABLED"              TINYINT       NOT NULL DEFAULT 0,
    "DEPLOYED"              TINYINT       NOT NULL DEFAULT 0, /* boolean */
    "IS_WATCHER"            TINYINT       NOT NULL DEFAULT 0,
    "OS_ID"                 INT  UNSIGNED NOT NULL,
    "TITLE"                 VARCHAR(255)  NULL,
    "VERSION"               VARCHAR(30)   NULL,
    "JAVA_VERSION"          VARCHAR(30)   NULL,
    "CERTIFICATE"           VARCHAR(4000) NULL,
    "MODIFIED"              DATETIME      NOT NULL,    
    CONSTRAINT UNIQUE_ISI_S UNIQUE ("SUB_AGENT_ID"),
    PRIMARY KEY ("ID")
) ENGINE=InnoDB;

/* Table INV_SUBAGENT_CLUSTERS */
CREATE TABLE IF NOT EXISTS INV_SUBAGENT_CLUSTERS (
    "ID"                    INT           NOT NULL AUTO_INCREMENT,
    "AGENT_ID"              VARCHAR(255)  NOT NULL,
    "SUBAGENT_CLUSTER_ID"   VARCHAR(255)  NOT NULL,
    "TITLE"                 VARCHAR(255)  NULL,
    "DEPLOYED"              TINYINT       NOT NULL, /* boolean */
    "ORDERING"              INT           NOT NULL DEFAULT 0,
    "MODIFIED"              DATETIME      NOT NULL,    
    CONSTRAINT UNIQUE_ISC_AC UNIQUE ("SUBAGENT_CLUSTER_ID"),
    PRIMARY KEY ("ID")
) ENGINE=InnoDB;

/* Table INV_SUBAGENT_CLUSTER_MEMBERS */
CREATE TABLE IF NOT EXISTS INV_SUBAGENT_CLUSTER_MEMBERS (
    "ID"                    INT           NOT NULL AUTO_INCREMENT,
    "SUBAGENT_CLUSTER_ID"   VARCHAR(255)  NOT NULL,
    "SUBAGENT_ID"           VARCHAR(255)  NOT NULL,
    "PRIORITY"              VARCHAR(255)  NOT NULL,
    "MODIFIED"              DATETIME      NOT NULL,    
    CONSTRAINT UNIQUE_ISCM_CS UNIQUE ("SUBAGENT_CLUSTER_ID", "SUBAGENT_ID"),
    PRIMARY KEY ("ID")
) ENGINE=InnoDB;

/* Table INV_CONFIGURATIONS */
CREATE TABLE IF NOT EXISTS INV_CONFIGURATIONS(
    "ID"                    INT        UNSIGNED NOT NULL AUTO_INCREMENT,
    "TYPE"                  TINYINT(2) UNSIGNED NOT NULL,   /* com.sos.joc.model.inventory.ConfigurationType */
    "PATH"                  VARCHAR(255)        NOT NULL,
    "NAME"                  VARCHAR(255)        NOT NULL,
    "FOLDER"                VARCHAR(255)        NOT NULL,
    "TITLE"                 VARCHAR(255)        NULL,
    "CONTENT"               LONGTEXT            NULL,
    "JSON_CONTENT"          JSON                NULL,
    "VALID"                 TINYINT(1) UNSIGNED NOT NULL, /* boolean */
    "DELETED"               TINYINT(1) UNSIGNED NOT NULL, /* boolean */
    "DEPLOYED"              TINYINT(1) UNSIGNED NOT NULL, /* boolean */
    "RELEASED"              TINYINT(1) UNSIGNED NOT NULL, /* boolean */
    "REPO_CTRL"             TINYINT(1) UNSIGNED NOT NULL, /* boolean */
    "AUDIT_LOG_ID"          INT        UNSIGNED NOT NULL,
    "CREATED"               DATETIME            NOT NULL,
    "MODIFIED"              DATETIME            NOT NULL,
    INDEX IDX_IC_F("FOLDER"),
    CONSTRAINT UNIQUE_IC_TP UNIQUE ("TYPE", "PATH"),
    PRIMARY KEY ("ID")
) ENGINE=InnoDB;

/* Table INV_CONFIGURATION_TRASH */
CREATE TABLE IF NOT EXISTS INV_CONFIGURATION_TRASH(
    "ID"                    INT        UNSIGNED NOT NULL AUTO_INCREMENT,
    "TYPE"                  TINYINT(2) UNSIGNED NOT NULL,   /* com.sos.joc.model.inventory.ConfigurationType */
    "PATH"                  VARCHAR(255)        NOT NULL,
    "NAME"                  VARCHAR(255)        NOT NULL,
    "FOLDER"                VARCHAR(255)        NOT NULL,
    "TITLE"                 VARCHAR(255)        NULL,
    "CONTENT"               LONGTEXT            NULL,
    "VALID"                 TINYINT(1) UNSIGNED NOT NULL, /* boolean */
    "AUDIT_LOG_ID"          INT        UNSIGNED NOT NULL,
    "CREATED"               DATETIME            NOT NULL,
    "MODIFIED"              DATETIME            NOT NULL,
    INDEX IDX_ICT_F("FOLDER"),
    CONSTRAINT UNIQUE_ICT_TP UNIQUE ("TYPE", "PATH"),
    PRIMARY KEY ("ID")
) ENGINE=InnoDB;

/* Table INV_RELEASED_CONFIGURATIONS */
CREATE TABLE IF NOT EXISTS INV_RELEASED_CONFIGURATIONS(
    "ID"                    INT        UNSIGNED NOT NULL AUTO_INCREMENT,
    "TYPE"                  TINYINT(2) UNSIGNED NOT NULL,   /* com.sos.joc.model.inventory.ConfigurationType */
    "PATH"                  VARCHAR(255)        NOT NULL,
    "CID"                   INT        UNSIGNED NOT NULL, /* 1:1 foreign key INV_CONFIGURATIONS.ID */
    "NAME"                  VARCHAR(255)        NOT NULL,
    "FOLDER"                VARCHAR(255)        NOT NULL,
    "TITLE"                 VARCHAR(255)        NULL,
    "CONTENT"               LONGTEXT            NOT NULL,
    "JSON_CONTENT"          JSON                NOT NULL,
    "AUDIT_LOG_ID"          INT        UNSIGNED NOT NULL,
    "CREATED"               DATETIME            NOT NULL,
    "MODIFIED"              DATETIME            NOT NULL,
    INDEX IDX_IRC_C("CID"),
    CONSTRAINT UNIQUE_IRC_TP UNIQUE ("TYPE", "PATH"),
    PRIMARY KEY ("ID")
) ENGINE=InnoDB;

/* Table INV_CERTS*/
CREATE TABLE IF NOT EXISTS INV_CERTS(
    "ID"                    INT             UNSIGNED NOT NULL AUTO_INCREMENT,
    "KEY_TYPE"              TINYINT(1)      UNSIGNED NOT NULL, /* PRIVATE, X509 */ 
    "KEY_ALG"               TINYINT(1)      UNSIGNED NOT NULL, /* RSA, ECDSA */
    "PEM"                   VARCHAR(4000)            NOT NULL,
    "CA"                    TINYINT(1)      UNSIGNED NOT NULL, /* boolean */
    "ACCOUNT"               VARCHAR(255)             NOT NULL,
    "SECLVL"                TINYINT         UNSIGNED NOT NULL DEFAULT 0,
    CONSTRAINT UNIQUE_ICS_KTCAS UNIQUE ("KEY_TYPE", "CA", "ACCOUNT", "SECLVL"),
    PRIMARY KEY ("ID")
) ENGINE=InnoDB;

/* Table for DOCUMENTATIONS */
CREATE TABLE IF NOT EXISTS INV_DOCUMENTATIONS(
    "ID"                    INT  UNSIGNED NOT NULL AUTO_INCREMENT,
    "NAME"                  VARCHAR(255)  NOT NULL,
    "FOLDER"                VARCHAR(255)  NOT NULL,
    "PATH"                  VARCHAR(255)  NOT NULL,
    "TYPE"                  VARCHAR(25)   NOT NULL DEFAULT '',
    "CONTENT"               LONGTEXT      NULL,
    "IMAGE_ID"              INT           NULL,
    "IS_REF"                TINYINT(1)    NOT NULL DEFAULT 0,
    "DOC_REF"               VARCHAR(255)  NULL,
    "CREATED"               DATETIME      NOT NULL,
    "MODIFIED"              DATETIME      NOT NULL,
    INDEX IDX_ID_IR("IS_REF"),
    CONSTRAINT UNIQUE_IDOC_PATH UNIQUE ("PATH"),
    PRIMARY KEY ("ID")
) ENGINE=InnoDB;

/* DOCUMENTATION_IMAGES */
CREATE TABLE IF NOT EXISTS INV_DOCUMENTATION_IMAGES(
    "ID"                    INT    UNSIGNED NOT NULL AUTO_INCREMENT,
    "IMAGE"                 MEDIUMBLOB      NOT NULL,
    "MD5_HASH"              CHAR(32)        NOT NULL,
    PRIMARY KEY ("ID")
) ENGINE=InnoDB;

/* INV_SEARCH_ITEMS */
/* Help table (see inventory_schedule2workflows_procedure.sql, inventory_schedule2calendars_procedure.sql) - solution for MySQL 5.7 to extract JSON array values into multiple rows */
CREATE TABLE IF NOT EXISTS INV_SEARCH_ITEMS (
  "ROW" INT(10) UNSIGNED NOT NULL,
  PRIMARY KEY ("ROW")
) ENGINE=InnoDB;

/* Table INV_FAVORITES */
CREATE TABLE IF NOT EXISTS INV_FAVORITES (
    "ID"                    INT  UNSIGNED NOT NULL AUTO_INCREMENT,
    "TYPE"                  TINYINT(2) UNSIGNED NOT NULL,   /* e.g. 1=facet, 2=agent */
    "NAME"                  VARCHAR(255)  NOT NULL,
    "FAVORITE"              LONGTEXT      NULL,
    "ORDERING"              INT           NOT NULL DEFAULT 0,
    "ACCOUNT"               VARCHAR(255)  NOT NULL,
    "SHARED"                TINYINT(1)    NOT NULL DEFAULT 0,
    "CREATED"               DATETIME      NOT NULL,
    "MODIFIED"              DATETIME      NOT NULL,
    CONSTRAINT UNIQUE_IF_TNA UNIQUE ("TYPE","NAME","ACCOUNT"),
    PRIMARY KEY ("ID")
) ENGINE=InnoDB;

/* Table INV_TAGS */
CREATE TABLE IF NOT EXISTS INV_TAGS (
    "ID"                    INT  UNSIGNED NOT NULL AUTO_INCREMENT,
    "NAME"                  VARCHAR(255)  NOT NULL,
    "ORDERING"              INT           NOT NULL DEFAULT 0,
    "GROUP_ID"              INT  UNSIGNED NOT NULL DEFAULT 0,
    "MODIFIED"              DATETIME      NOT NULL,
    CONSTRAINT UNIQUE_IT_N UNIQUE ("NAME"),
    PRIMARY KEY ("ID")
) ENGINE=InnoDB;

/* Table INV_TAGGINGS */
CREATE TABLE IF NOT EXISTS INV_TAGGINGS (
    "ID"                    INT  UNSIGNED NOT NULL AUTO_INCREMENT,
    "CID"              	    INT  UNSIGNED NOT NULL,
    "NAME"                  VARCHAR(255)  NOT NULL,
    "TYPE"                  TINYINT(2) UNSIGNED NOT NULL, /* com.sos.joc.model.inventory.ConfigurationType */
    "TAG_ID"              	INT  UNSIGNED NOT NULL,
    "MODIFIED"              DATETIME      NOT NULL,
    CONSTRAINT UNIQUE_ITG_CT UNIQUE ("CID","TAG_ID"),
    PRIMARY KEY ("ID")
) ENGINE=InnoDB;

/* Table INV_JOB_TAGS */
CREATE TABLE IF NOT EXISTS INV_JOB_TAGS (
    "ID"                    INT  UNSIGNED NOT NULL AUTO_INCREMENT,
    "NAME"                  VARCHAR(255)  NOT NULL,
    "ORDERING"              INT           NOT NULL DEFAULT 0,
    "GROUP_ID"              INT  UNSIGNED NOT NULL DEFAULT 0,
    "MODIFIED"              DATETIME      NOT NULL,
    CONSTRAINT UNIQUE_IJT_N UNIQUE ("NAME"),
    PRIMARY KEY ("ID")
) ENGINE=InnoDB;

/* Table INV_JOB_TAGGINGS */
CREATE TABLE IF NOT EXISTS INV_JOB_TAGGINGS (
    "ID"                    INT  UNSIGNED NOT NULL AUTO_INCREMENT,
    "CID"              	    INT  UNSIGNED NOT NULL,
    "WORKFLOW_NAME"         VARCHAR(255)  NOT NULL,
    "JOB_NAME"              VARCHAR(255)  NOT NULL,
    "TAG_ID"              	INT  UNSIGNED NOT NULL,
    "MODIFIED"              DATETIME      NOT NULL,
    CONSTRAINT UNIQUE_IJTG_CJT UNIQUE ("CID", "JOB_NAME", "TAG_ID"),
    PRIMARY KEY ("ID")
) ENGINE=InnoDB;

/* Table INV_ADD_ORDER_TAGS */
CREATE TABLE IF NOT EXISTS INV_ADD_ORDER_TAGS (
    "ORDER_ID_PATTERN"      BIGINT  UNSIGNED NOT NULL,
    "ORDER_TAGS"            LONGTEXT         NOT NULL,
    PRIMARY KEY ("ORDER_ID_PATTERN")
) ENGINE=InnoDB;

/* Table INV_ORDER_TAGS */
CREATE TABLE IF NOT EXISTS INV_ORDER_TAGS (
    "ID"                    INT  UNSIGNED NOT NULL AUTO_INCREMENT,
    "NAME"                  VARCHAR(255)  NOT NULL,
    "ORDERING"              INT           NOT NULL DEFAULT 0,
    "GROUP_ID"              INT  UNSIGNED NOT NULL DEFAULT 0,
    "MODIFIED"              DATETIME      NOT NULL,
    CONSTRAINT UNIQUE_IOT_N UNIQUE ("NAME"),
    PRIMARY KEY ("ID")
) ENGINE=InnoDB;

/* Table INV_TAG_GROUPS */
CREATE TABLE IF NOT EXISTS INV_TAG_GROUPS (
    "ID"                    INT  UNSIGNED NOT NULL AUTO_INCREMENT,
    "NAME"                  VARCHAR(255)  NOT NULL,
    "ORDERING"              INT           NOT NULL DEFAULT 0,
    "MODIFIED"              DATETIME      NOT NULL,
    CONSTRAINT UNIQUE_ITG_N UNIQUE ("NAME"),
    PRIMARY KEY ("ID")
) ENGINE=InnoDB;

/* Table INV_DEPENDENCIES */
CREATE TABLE IF NOT EXISTS INV_DEPENDENCIES (
    "INV_ID"                INT           UNSIGNED NOT NULL,
    "INV_DEP_ID"            INT           UNSIGNED NOT NULL,
    "DEP_TYPE"              TINYINT(2)    UNSIGNED NOT NULL, /* com.sos.joc.model.inventory.ConfigurationType */
    "PUBLISHED"             TINYINT(1)    UNSIGNED NOT NULL DEFAULT 0,
    "DEP_DEP_ID"            INT           UNSIGNED NULL,
    "CONTROLLER_ID"         VARCHAR(100)  NULL,
    PRIMARY KEY ("INV_ID", "INV_DEP_ID")
) ENGINE=InnoDB;

/* Table INV_CHANGES */
CREATE TABLE IF NOT EXISTS INV_CHANGES (
    "ID"                    INT           UNSIGNED NOT NULL AUTO_INCREMENT,
    "NAME"                  VARCHAR(100)  NOT NULL,
    "TITLE"                 VARCHAR(255)  NULL,
    "STATE"                 TINYINT(2)    UNSIGNED NOT NULL,
    "CREATED"               DATETIME      NOT NULL,
    "MODIFIED"              DATETIME      NOT NULL,
    "CLOSED"                DATETIME      NULL,
    "OWNER"                 VARCHAR(255)  NOT NULL,
    "LAST_PUBLISHED_BY"     VARCHAR(255)  NULL,
    INDEX IDX_INV_CH_ST("STATE"),
    CONSTRAINT UNIQUE_ICH_N UNIQUE ("NAME"),
    PRIMARY KEY ("ID")
) ENGINE=InnoDB;

/* Table INV_CHANGES_MAPPINGS */
CREATE TABLE IF NOT EXISTS INV_CHANGES_MAPPINGS (
    "CHANGE_ID"             INT           UNSIGNED NOT NULL,
    "INV_ID"                INT           UNSIGNED NOT NULL,
    "TYPE"                  TINYINT(2)    UNSIGNED NOT NULL,   /* com.sos.joc.model.inventory.ConfigurationType */
    PRIMARY KEY ("CHANGE_ID", "INV_ID")
) ENGINE=InnoDB;

/*
 | instances table for MySQL
*/
set SQL_MODE=ANSI_QUOTES;

/* Table for VARIABLES */
CREATE TABLE IF NOT EXISTS JOC_VARIABLES(
    "NAME"                  VARCHAR(255)  NOT NULL,
    "NUMERIC_VALUE"         INT           NULL,
    "TEXT_VALUE"            VARCHAR(255)  NULL,
    "BINARY_VALUE"          LONGBLOB      NULL,
    PRIMARY KEY ("NAME")
) ENGINE=InnoDB;

/* Table JOC_INSTANCES */
CREATE TABLE IF NOT EXISTS JOC_INSTANCES (
    "ID"                    INT  UNSIGNED NOT NULL AUTO_INCREMENT,
    "CLUSTER_ID"            VARCHAR(10)  NOT NULL,
    "MEMBER_ID"             VARCHAR(165)  NOT NULL, /* HOSTNAME:hash(DATA_DIRECTORY) */
    "OS_ID"                 INT  UNSIGNED NOT NULL,
    "DATA_DIRECTORY"        VARCHAR(255)  NOT NULL,
    "SECURITY_LEVEL"        VARCHAR(10)   NOT NULL,
    "TIMEZONE"              VARCHAR(30)   NOT NULL,
    "STARTED_AT"            DATETIME      NOT NULL,
    "TITLE"                 VARCHAR(30)   NULL,
    "ORDERING"              TINYINT       NOT NULL DEFAULT 0,
    "URI"                   VARCHAR(255)  NULL,
    "HEART_BEAT"            DATETIME      NOT NULL,
    "API_SERVER"            TINYINT(1)  UNSIGNED  NOT NULL,
    "VERSION"               VARCHAR(30)   NULL,
    "CERTIFICATE"           VARCHAR(4000) NULL,           /* X.509 Certificate*/
    CONSTRAINT UNIQUE_JI_MID UNIQUE ("MEMBER_ID"),
    PRIMARY KEY ("ID")
) ENGINE=InnoDB;

/* Table JOC_CLUSTER */
CREATE TABLE IF NOT EXISTS JOC_CLUSTER (
    "ID"                        VARCHAR(10)   NOT NULL,
    "MEMBER_ID"                 VARCHAR(165)  NOT NULL,
    "HEART_BEAT"                DATETIME      NOT NULL,
    "SWITCH_MEMBER_ID"          VARCHAR(165),
    "SWITCH_HEART_BEAT"         DATETIME,
    "NOTIFICATION_MEMBER_ID"    VARCHAR(165),
    "NOTIFICATION"              VARCHAR(100),
    CONSTRAINT UNIQUE_JC_MID UNIQUE ("MEMBER_ID"),
    PRIMARY KEY ("ID")
) ENGINE=InnoDB;

/* Table JOC_CONFIGURATIONS */
CREATE TABLE IF NOT EXISTS JOC_CONFIGURATIONS (
    "ID"                    INT  UNSIGNED NOT NULL AUTO_INCREMENT,
    "INSTANCE_ID"           INT  UNSIGNED NULL,
    "CONTROLLER_ID"         VARCHAR(100)  NULL,
    "ACCOUNT"               VARCHAR(255)  NOT NULL,
    "OBJECT_TYPE"           VARCHAR(30)   NULL,
    "CONFIGURATION_TYPE"    VARCHAR(30)   NOT NULL,
    "NAME"                  VARCHAR(255)  NULL,
    "SHARED"                TINYINT       NOT NULL DEFAULT 0,
    "CONFIGURATION_ITEM"    LONGTEXT      NOT NULL,
    "MODIFIED"              DATETIME      NOT NULL,
    CONSTRAINT UNIQUE_JC_CIDAOTCTN UNIQUE ("CONTROLLER_ID","ACCOUNT","OBJECT_TYPE","CONFIGURATION_TYPE","NAME"),
    PRIMARY KEY ("ID")
) ENGINE=InnoDB;

/* Table JOC_AUDIT_LOG */
CREATE TABLE IF NOT EXISTS JOC_AUDIT_LOG (
    "ID"                    INT UNSIGNED  NOT NULL AUTO_INCREMENT,
    "CONTROLLER_ID"         VARCHAR(100)  NOT NULL,
    "ACCOUNT"               VARCHAR(255)  NOT NULL,
    "REQUEST"               VARCHAR(50)   NOT NULL,
    "PARAMETERS"            LONGTEXT      NULL,
    "CATEGORY"              TINYINT(2) UNSIGNED NOT NULL DEFAULT 0,
    "COMMENT"               VARCHAR(2000) NULL,
    "CREATED"               DATETIME      NOT NULL,
    "TICKET_LINK"           VARCHAR(255)  NULL,
    "TIME_SPENT"            INT(10)       NULL,
    INDEX IDX_JAL_CCC("CREATED","CONTROLLER_ID","CATEGORY"),
    PRIMARY KEY ("ID")
) ENGINE=InnoDB;

/* Table JOC_AUDIT_LOG_DETAILS */
CREATE TABLE IF NOT EXISTS JOC_AUDIT_LOG_DETAILS (
    "ID"                    INT UNSIGNED        NOT NULL AUTO_INCREMENT,
    "AUDITLOG_ID"           INT UNSIGNED        NOT NULL, /* ID from JOC_AUDIT_LOG */
    "TYPE"                  TINYINT(2) UNSIGNED NOT NULL DEFAULT 0, 
    "FOLDER"                VARCHAR(255)        NOT NULL,
    "PATH"                  VARCHAR(255)        NOT NULL,
    "NAME"                  VARCHAR(255)        NOT NULL,
    "ORDER_ID"              VARCHAR(255)        NULL, /*For an order the workflow name fills the NAME field */
    "CREATED"               DATETIME            NOT NULL,
    INDEX IDX_JALD_JALID("AUDITLOG_ID"),
    INDEX IDX_JALD_FOLDER("FOLDER"),
    INDEX IDX_JALD_TYPE("TYPE"),
    PRIMARY KEY ("ID")
) ENGINE=InnoDB;


/* Table JOC_LOCKS */
CREATE TABLE IF NOT EXISTS JOC_LOCKS (
    "ID"                    INT     UNSIGNED    NOT NULL    AUTO_INCREMENT,
    "RANGE"                 TINYINT UNSIGNED    NOT NULL,
    "FOLDER"                VARCHAR(255)        NOT NULL,
    "ACCOUNT"               VARCHAR(255)        NOT NULL,
    "CREATED"               DATETIME      NOT NULL,
    CONSTRAINT UNIQUE_JL_RF UNIQUE ("RANGE", "FOLDER"),
    PRIMARY KEY ("ID")
) ENGINE=InnoDB;
/*
 | Monitoring Interface tables for MySQL
*/
set SQL_MODE=ANSI_QUOTES;

/* Table for MON_ORDERS */
CREATE TABLE IF NOT EXISTS MON_ORDERS(
    "HISTORY_ID"                BIGINT(20)      UNSIGNED    NOT NULL,   /* HISTORY_ORDERS.ID */
    "CONTROLLER_ID"             VARCHAR(100)                NOT NULL,
    "ORDER_ID"                  VARCHAR(255)                NOT NULL,
    "WORKFLOW_PATH"             VARCHAR(255)                NOT NULL,
    "WORKFLOW_VERSION_ID"       VARCHAR(255)                NOT NULL,   /* #2019-06-13T08:43:29Z */
    "WORKFLOW_POSITION"         VARCHAR(255)                NOT NULL,   /* 1#fork_1#0 */
    "WORKFLOW_FOLDER"           VARCHAR(255)                NOT NULL,
    "WORKFLOW_NAME"             VARCHAR(255)                NOT NULL,
    "WORKFLOW_TITLE"            VARCHAR(255),                           
    "MAIN_PARENT_ID"            BIGINT(20)      UNSIGNED    NOT NULL,   /* HISTORY_ORDERS.ID of the main order */
    "PARENT_ID"                 BIGINT(20)      UNSIGNED    NOT NULL,   /* HISTORY_ORDERS.ID of the parent order */
    "PARENT_ORDER_ID"           VARCHAR(255)                NOT NULL,   /* HISTORY_ORDERS.ORDER_ID */
    "HAS_CHILDREN"              TINYINT(1)      UNSIGNED    NOT NULL,
    "NAME"                      VARCHAR(255)                NOT NULL,   /* orderId or branchId by fork */
    "START_CAUSE"               VARCHAR(50)                 NOT NULL,   /* order, fork, file_trigger, setback, unskip, unstop */
    "START_TIME_SCHEDULED"      DATETIME,                               
    "START_TIME"                DATETIME                    NOT NULL,
    "START_WORKFLOW_POSITION"   VARCHAR(255)                NOT NULL,
    "START_VARIABLES"           LONGTEXT,                          
    "CURRENT_HOS_ID"            BIGINT(20)      UNSIGNED    NOT NULL,   /* HISTORY_ORDER_STEPS.ID */
    "END_TIME"                  DATETIME,
    "END_WORKFLOW_POSITION"     VARCHAR(255),
    "END_HOS_ID"                BIGINT(20)      UNSIGNED    NOT NULL,   /* HISTORY_ORDER_STEPS.ID */
    "END_RETURN_CODE"           INT(10),
    "END_MESSAGE"               VARCHAR(500),
    "SEVERITY"                  TINYINT(1)      UNSIGNED    NOT NULL,   /* 0,1,2 */
    "STATE"                     TINYINT         UNSIGNED    NOT NULL,   
    "STATE_TIME"                DATETIME                    NOT NULL,
    "ERROR"                     TINYINT(1)      UNSIGNED    NOT NULL,
    "ERROR_STATE"               VARCHAR(20),                            /* failed, disrupted ... - event outcome*/
    "ERROR_REASON"              VARCHAR(50),                            /* other ... - event outcome*/
    "ERROR_RETURN_CODE"         INT(10),
    "ERROR_CODE"                VARCHAR(50),                            
    "ERROR_TEXT"                VARCHAR(500),                           
    "LOG_ID"                    BIGINT(20)      UNSIGNED,               /* HISTORY_LOGS.ID */
    "CREATED"                   DATETIME                    NOT NULL,
    "MODIFIED"                  DATETIME                    NOT NULL,
    INDEX IDX_MONO_CID("CONTROLLER_ID"),
    INDEX IDX_MONO_MPID("MAIN_PARENT_ID"),
    PRIMARY KEY ("HISTORY_ID")
) ENGINE=InnoDB;

/* Table for MON_ORDER_STEPS */
CREATE TABLE IF NOT EXISTS MON_ORDER_STEPS(
    "HISTORY_ID"                BIGINT(20)      UNSIGNED    NOT NULL,   /* HISTORY_ORDER_STEPS.ID */
    "WORKFLOW_POSITION"         VARCHAR(255)                NOT NULL,   /* 1#fork_1#3 */
    "HO_MAIN_PARENT_ID"         BIGINT(20)      UNSIGNED    NOT NULL,   /* HISTORY_ORDERS.MAIN_PARENT_ID */
    "HO_ID"                     BIGINT(20)      UNSIGNED    NOT NULL,   /* HISTORY_ORDERS.ID */
    "POSITION"                  INT             UNSIGNED    NOT NULL,   /* 3 - last position from WORKFLOW_POSITION */
    "JOB_NAME"                  VARCHAR(255)                NOT NULL,
    "JOB_LABEL"                 VARCHAR(255)                NOT NULL,
    "JOB_TITLE"                 VARCHAR(255),                           
    "JOB_NOTIFICATION"          VARCHAR(1000),
    "JOB_CRITICALITY"           TINYINT         UNSIGNED    NOT NULL,
    "AGENT_ID"                  VARCHAR(255)                NOT NULL,
    "AGENT_NAME"                VARCHAR(255),
    "AGENT_URI"                 VARCHAR(255)                NOT NULL,
    "SUBAGENT_CLUSTER_ID"       VARCHAR(255),
    "START_CAUSE"               VARCHAR(50)                 NOT NULL,   /* order, file_trigger, setback, unskip, unstop */
    "START_TIME"                DATETIME                    NOT NULL,
    "START_VARIABLES"           LONGTEXT,                          
    "END_TIME"                  DATETIME,
    "END_VARIABLES"             LONGTEXT,                          
    "RETURN_CODE"               INT(10),
    "SEVERITY"                  TINYINT(1)      UNSIGNED    NOT NULL,   /* 0,1,2 */
    "ERROR"                     TINYINT(1)      UNSIGNED    NOT NULL,   
    "ERROR_STATE"               VARCHAR(20),                            /* failed, disrupted ... - event outcome*/
    "ERROR_REASON"              VARCHAR(50),                            /* other ... - event outcome*/
    "ERROR_CODE"                VARCHAR(50),                            
    "ERROR_TEXT"                VARCHAR(500),                           
    "LOG_ID"                    BIGINT(20)      UNSIGNED,               /* HISTORY_LOGS.ID */
    "CREATED"                   DATETIME                    NOT NULL,
    "MODIFIED"                  DATETIME                    NOT NULL,
    INDEX IDX_MONOS_HMPID("HO_MAIN_PARENT_ID"),
    PRIMARY KEY ("HISTORY_ID")
) ENGINE=InnoDB;


/* Table for MON_NOTIFICATIONS */
CREATE TABLE IF NOT EXISTS MON_NOTIFICATIONS(
    "ID"                        BIGINT(20)      UNSIGNED    NOT NULL AUTO_INCREMENT,
    "TYPE"                      TINYINT         UNSIGNED    NOT NULL,   /* SUCCESS(0), ERROR(1), WARNING(2), RECOVERED(3), ACKNOWLEDGED(4) */
    "RANGE"                     TINYINT         UNSIGNED    NOT NULL,   /* WORKFLOW(0), WORKFLOW_JOB(1) */
    "NOTIFICATION_ID"           VARCHAR(255)                NOT NULL,
    "RECOVERED_ID"              BIGINT(20)      UNSIGNED    NOT NULL,   /* MON_NOTIFICATIONS.ID */
    "WARN"                      TINYINT         UNSIGNED    NOT NULL,  
    "WARN_TEXT"                 VARCHAR(500),                           
    "HAS_MONITORS"              TINYINT(1)      UNSIGNED    NOT NULL,
    "CREATED"                   DATETIME                    NOT NULL,
    INDEX IDX_MONN_C("CREATED"),
    PRIMARY KEY ("ID")
) ENGINE=InnoDB;

/* Table for MON_NOT_WORKFLOWS */
CREATE TABLE IF NOT EXISTS MON_NOT_WORKFLOWS(
    "NOT_ID"                    BIGINT(20)      UNSIGNED    NOT NULL,   /* MON_NOTIFICATIONS.ID */
    "MON_O_HISTORY_ID"          BIGINT(20)      UNSIGNED    NOT NULL,   /* MON_ORDERS.HISTORY_ID */
    "MON_OS_HISTORY_ID"         BIGINT(20)      UNSIGNED    NOT NULL,   /* MON_ORDER_STEPS.HISTORY_ID */
    "WORKFLOW_POSITION"         VARCHAR(255)                NOT NULL,   /* 1#fork_1#3 */
    INDEX IDX_MONNW_OHID("MON_O_HISTORY_ID"),
    INDEX IDX_MONNW_OSHID("MON_OS_HISTORY_ID"),
    PRIMARY KEY ("NOT_ID")
) ENGINE=InnoDB;

/* Table for MON_SYSNOTIFICATIONS */
CREATE TABLE IF NOT EXISTS MON_SYSNOTIFICATIONS(
    "ID"                        BIGINT(20)      UNSIGNED    NOT NULL AUTO_INCREMENT,
    "TYPE"                      TINYINT         UNSIGNED    NOT NULL,   /* ERROR(1), WARNING(2), ACKNOWLEDGED(4) */
    "CATEGORY"                  TINYINT(1)      UNSIGNED    NOT NULL,   /* SYSTEM(0), JOC(1), CONTROLLER(2), AGENT(3) */
    "JOC_ID"                    VARCHAR(13)                 NOT NULL,   /* JOC_INSTANCES.CLUSTER_ID#JOC_INSTANCES.ORDERING */
    "SOURCE"                    VARCHAR(100)                NOT NULL,
    "NOTIFIER"                  VARCHAR(255)                NOT NULL,
    "TIME"                      DATETIME                    NOT NULL,
    "MESSAGE"                   VARCHAR(1000)               NULL,
    "EXCEPTION"                 VARCHAR(4000)               NULL,
    "HAS_MONITORS"              TINYINT(1)      UNSIGNED    NOT NULL,
    "CREATED"                   DATETIME                    NOT NULL,
    INDEX IDX_MONSN_TYPE("TYPE"),
    INDEX IDX_MONSN_JOCID("JOC_ID"),
    INDEX IDX_MONSN_TIME("TIME"),
    PRIMARY KEY ("ID")
) ENGINE=InnoDB;

/* Table for MON_NOT_MONITORS */
CREATE TABLE IF NOT EXISTS MON_NOT_MONITORS(
    "ID"                        BIGINT(20)      UNSIGNED    NOT NULL AUTO_INCREMENT,
    "NOT_ID"                    BIGINT(20)      UNSIGNED    NOT NULL,   /* MON_NOTIFICATIONS.ID or MON_SYSNOTIFICATIONS.ID  */
    "APPLICATION"               TINYINT(1)      UNSIGNED    NOT NULL,   /* 0 - MON_NOTIFICATIONS, 1 - MON_SYSNOTIFICATIONS */
    "TYPE"                      TINYINT         UNSIGNED    NOT NULL,   /* COMMAND(0), MAIL(1), NSCA(2), JMS(3) */
    "NAME"                      VARCHAR(255)                NOT NULL,
    "CONFIGURATION"             VARCHAR(500)                NOT NULL,
    "MESSAGE"                   VARCHAR(4000)               NOT NULL,
    "ERROR"                     TINYINT         UNSIGNED    NOT NULL,
    "ERROR_TEXT"                VARCHAR(500)                NULL,
    "CREATED"                   DATETIME                    NOT NULL,
    INDEX IDX_MONM_NIDA("NOT_ID","APPLICATION"),
    PRIMARY KEY ("ID")
) ENGINE=InnoDB;

/* Table for MON_NOT_ACKNOWLEDGEMENTS */
CREATE TABLE IF NOT EXISTS MON_NOT_ACKNOWLEDGEMENTS(
    "NOT_ID"                    BIGINT(20)      UNSIGNED    NOT NULL,   /* MON_NOTIFICATIONS.ID or MON_SYSNOTIFICATIONS.ID */
    "APPLICATION"               TINYINT(1)      UNSIGNED    NOT NULL,   /* 0 - MON_NOTIFICATIONS, 1 - MON_SYSNOTIFICATIONS */
    "ACCOUNT"                   VARCHAR(255)                NOT NULL,
    "COMMENT"                   VARCHAR(2000)               NULL,
    "CREATED"                   DATETIME                    NOT NULL,
    PRIMARY KEY ("NOT_ID","APPLICATION")
) ENGINE=InnoDB;
set SQL_MODE=ANSI_QUOTES;

/* MON_ORDER_STEPS columns */
START TRANSACTION; 
set @exist := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'MON_ORDER_STEPS' and column_name = 'START_VARIABLES' and upper(column_type) = 'LONGTEXT' and table_schema = database());
set @sqlstatement := if( @exist < 1, 'ALTER TABLE MON_ORDER_STEPS MODIFY COLUMN "START_VARIABLES" LONGTEXT NULL;', 'select ''INFO: START_VARIABLES LONGTEXT found''');
PREPARE statement FROM @sqlstatement;
EXECUTE statement;
COMMIT;
/*
 | tables for MySQL
*/
set SQL_MODE=ANSI_QUOTES;

/* Table for REPORTS */
CREATE TABLE IF NOT EXISTS REPORTS (
    "ID"                    BIGINT  UNSIGNED NOT NULL AUTO_INCREMENT,
    "RUN_ID"                BIGINT  UNSIGNED NOT NULL, /* REPORT_RUNS.id */
    "FREQUENCY"             INT     UNSIGNED NOT NULL,
    "DATE_FROM"             DATETIME         NOT NULL,
    "DATE_TO"               DATETIME         NOT NULL,
    "CONTENT"               LONGTEXT         NOT NULL,
    "CONSTRAINT_HASH"       CHAR(64)         NOT NULL,
    "MODIFIED"              DATETIME         NOT NULL,
    "CREATED"               DATETIME         NOT NULL,
    CONSTRAINT UNIQUE_R_CH UNIQUE ("CONSTRAINT_HASH"),
    PRIMARY KEY ("ID")
) ENGINE=InnoDB;

/* Table for REPORT_RUNS */
CREATE TABLE IF NOT EXISTS REPORT_RUNS (
    "ID"                    BIGINT  UNSIGNED NOT NULL AUTO_INCREMENT,
    "PATH"                  VARCHAR(255)     NOT NULL,
    "FOLDER"                VARCHAR(255)     NOT NULL,
    "NAME"                  VARCHAR(255)     NOT NULL,
    "TITLE"                 VARCHAR(255)     NULL,
    "TEMPLATE_ID"           INT     UNSIGNED NOT NULL,
    "FREQUENCIES"           VARCHAR(255)     NOT NULL,
    "SORT"                  INT     UNSIGNED NOT NULL,
    "PERIOD_LENGTH"         INT     UNSIGNED NULL,
    "PERIOD_STEP"           INT     UNSIGNED NULL,
    "HITS"                  INT     UNSIGNED NOT NULL,
    "REPORT_COUNT"          INT     UNSIGNED NOT NULL,
    "CONTROLLER_ID"         VARCHAR(100)     NULL,
    "MONTH_FROM"            DATETIME         NOT NULL,
    "MONTH_TO"              DATETIME         NOT NULL,
    "STATE"                 TINYINT UNSIGNED NOT NULL,
    "ERROR_TEXT"            VARCHAR(500)     NULL,
    "MODIFIED"              DATETIME         NOT NULL,
    "CREATED"               DATETIME         NOT NULL,
    PRIMARY KEY ("ID")
) ENGINE=InnoDB;

 
/*
 | Job Scheduler Inventory search tables for MySQL
 | SOS GmbH, 2020-10-29
*/
set SQL_MODE=ANSI_QUOTES;

/* Table SEARCH_WORKFLOWS */
CREATE TABLE IF NOT EXISTS SEARCH_WORKFLOWS(
    "ID"                    INT        UNSIGNED NOT NULL AUTO_INCREMENT,
    "INV_CID"               INT        UNSIGNED NOT NULL, /* INV_CONFIGURATIONS.ID */
    "DEPLOYED"              TINYINT(1) UNSIGNED NOT NULL, /* boolean */
    "CONTENT_HASH"          CHAR(64)            NOT NULL, /* workflow hash */
    "JOBS_COUNT"            INT        UNSIGNED NOT NULL,
    "JOBS"                  JSON                NOT NULL,
    "ARGS"                  JSON                NOT NULL,
    "JOBS_SCRIPTS"          JSON                NOT NULL,
    "INSTRUCTIONS"          JSON                NOT NULL,
    "INSTRUCTIONS_ARGS"     JSON                NOT NULL,
    "CREATED"               DATETIME            NOT NULL,
    "MODIFIED"              DATETIME            NOT NULL,
    INDEX IDX_SW_INVCID("INV_CID"),
    INDEX IDX_SW_DEP("DEPLOYED"),
    INDEX IDX_SW_CH("CONTENT_HASH"),
    INDEX IDX_SW_JC("JOBS_COUNT"),
    PRIMARY KEY ("ID")
) ENGINE=InnoDB;

/* Table SEARCH_WORKFLOWS_DEP_H */
CREATE TABLE IF NOT EXISTS SEARCH_WORKFLOWS_DEP_H(
    "SEARCH_WID"            INT        UNSIGNED NOT NULL, /* SEARCH_WORKFLOWS.ID */
    "INV_CID"               INT        UNSIGNED NOT NULL, /* INV_CONFIGURATIONS.ID */
    "DEP_HID"               INT        UNSIGNED NOT NULL, /* DEP_HISTORY.ID */
    "CONTROLLER_ID"         VARCHAR(100)        NOT NULL,
    INDEX IDX_SWDH_DEPHID("DEP_HID"),
    PRIMARY KEY ("SEARCH_WID","INV_CID","CONTROLLER_ID")
) ENGINE=InnoDB;
set SQL_MODE=ANSI_QUOTES;

/* Table for XMLEDITOR_CONFIGURATIONS */
CREATE TABLE IF NOT EXISTS XMLEDITOR_CONFIGURATIONS (
    "ID"                            INT UNSIGNED  NOT NULL    AUTO_INCREMENT,
    "TYPE"                          VARCHAR(25)   NOT NULL,
    "NAME"                          VARCHAR(255)  NOT NULL,
    "SCHEMA_LOCATION"               VARCHAR(255)  NOT NULL,
    "CONFIGURATION_DRAFT"           LONGTEXT      NULL,
    "CONFIGURATION_DRAFT_JSON"      LONGTEXT      NULL,
    "CONFIGURATION_RELEASED"        LONGTEXT      NULL,
    "CONFIGURATION_RELEASED_JSON"   LONGTEXT      NULL,
    "AUDIT_LOG_ID"                  INT UNSIGNED  NOT NULL,
    "ACCOUNT"                       VARCHAR(255)  NOT NULL,
    "RELEASED"                      DATETIME,
    "MODIFIED"                      DATETIME      NOT NULL,
    "CREATED"                       DATETIME      NOT NULL,
    INDEX IDX_XEC_TN("TYPE", "NAME"),
    PRIMARY KEY ("ID")
) ENGINE=InnoDB;
/*
 | JobScheduler YADE History Interface tables for MySQL
 | SOS GmbH, 2021-03-24
*/
set SQL_MODE=ANSI_QUOTES;

/* Table for YADE_TRANSFERS */
CREATE TABLE IF NOT EXISTS YADE_TRANSFERS (
    "ID"                    BIGINT(20)    NOT NULL  AUTO_INCREMENT,
    "CONTROLLER_ID"         VARCHAR(100)  NOT NULL,
    "WORKFLOW_PATH"         VARCHAR(255)  NOT NULL,
    "WORKFLOW_NAME"         VARCHAR(255)  NOT NULL,
    "ORDER_ID"              VARCHAR(255)  NOT NULL,
    "JOB"                   VARCHAR(255)  NOT NULL,
    "JOB_POSITION"          VARCHAR(255)  NOT NULL,
    "HOS_ID"                BIGINT(20)    NOT NULL,
    "SOURCE_PROTOCOL_ID"    BIGINT(20)    NOT NULL,
    "TARGET_PROTOCOL_ID"    BIGINT(20)    NULL,
    "JUMP_PROTOCOL_ID"      BIGINT(20)    NULL,
    "OPERATION"             INT           NOT NULL,
    "PROFILE_NAME"          VARCHAR(100)  NULL,
    "START"                 DATETIME      NOT NULL,
    "END"                   DATETIME      NULL,
    "NUM_OF_FILES"          INT           NOT NULL,
    "STATE"                 INT           NOT NULL,
    "ERROR_MESSAGE"         VARCHAR(2000) NULL,
    "CREATED"               DATETIME      NOT NULL,
    INDEX IDX_YT_START("START"),
    PRIMARY KEY ("ID")
) ENGINE=InnoDB;

/* Table for YADE_PROTOCOLS */
CREATE TABLE IF NOT EXISTS YADE_PROTOCOLS (
    "ID"                    BIGINT(20)    NOT NULL  AUTO_INCREMENT,
    "HOSTNAME"              VARCHAR(255)  NOT NULL,
    "PORT"                  INT           NOT NULL,
    "PROTOCOL"              INT           NOT NULL,
    "ACCOUNT"               VARCHAR(255)  NOT NULL,
    "CREATED"               DATETIME      NOT NULL,
    CONSTRAINT UNIQUE_YP_HPPA UNIQUE ("HOSTNAME", "PORT", "PROTOCOL", "ACCOUNT"),
    PRIMARY KEY ("ID")
) ENGINE=InnoDB;

/* Table for YADE_FILES */
CREATE TABLE IF NOT EXISTS YADE_FILES (
    "ID"                    BIGINT(20)    NOT NULL  AUTO_INCREMENT,
    "TRANSFER_ID"           BIGINT(20)    NOT NULL,
    "SOURCE_PATH"           VARCHAR(255)  NOT NULL,
    "TARGET_PATH"           VARCHAR(255)  NULL,
    "SIZE"                  BIGINT        NULL,
    "MODIFICATION_DATE"     DATETIME      NULL,
    "STATE"                 INT           NOT NULL,
    "INTEGRITY_HASH"        VARCHAR(255)  NULL,
    "ERROR_MESSAGE"         VARCHAR(2000) NULL,
    "CREATED"               DATETIME      NOT NULL,
    CONSTRAINT INVENTORY_YF_UNIQUE UNIQUE ("TRANSFER_ID", "SOURCE_PATH"),
    PRIMARY KEY ("ID")
) ENGINE=InnoDB;
set SQL_MODE=ANSI_QUOTES;

/* IAM_ACCOUNTS columns */
START TRANSACTION; 
set @count_columns := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'IAM_ACCOUNTS' and column_name = 'FORCE_PASSWORD_CHANGE' and table_schema = database());
set @stmt := if( @count_columns < 1, 'ALTER TABLE IAM_ACCOUNTS ADD FORCE_PASSWORD_CHANGE TINYINT(4) NULL AFTER ACCOUNT_PASSWORD;', 'select ''INFO: IAM_ACCOUNTS not found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
set @stmt := if( @count_columns < 1, 'UPDATE IAM_ACCOUNTS SET "FORCE_PASSWORD_CHANGE"=0;', 'select ''INFO: UPDATE''');
PREPARE statement FROM @stmt;
EXECUTE statement;
set @stmt := if( @count_columns < 1, 'ALTER TABLE IAM_ACCOUNTS MODIFY FORCE_PASSWORD_CHANGE TINYINT(4) NOT NULL DEFAULT 0;', 'select ''INFO: NOT NULL''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

/* IAM_ACCOUNTS columns */
START TRANSACTION; 
set @count_columns := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'IAM_ACCOUNTS' and column_name = 'DISABLED' and table_schema = database());
set @stmt := if( @count_columns < 1, 'ALTER TABLE IAM_ACCOUNTS ADD DISABLED TINYINT(4) NULL AFTER FORCE_PASSWORD_CHANGE;', 'select ''INFO: IAM_ACCOUNTS not found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
set @stmt := if( @count_columns < 1, 'UPDATE IAM_ACCOUNTS SET "DISABLED"=0;', 'select ''INFO: UPDATE''');
PREPARE statement FROM @stmt;
EXECUTE statement;
set @stmt := if( @count_columns < 1, 'ALTER TABLE IAM_ACCOUNTS MODIFY DISABLED TINYINT(4) NOT NULL DEFAULT 0;', 'select ''INFO: NOT NULL''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

/* IAM_HISTORY columns */
START TRANSACTION; 
set @count_columns := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'IAM_HISTORY' and column_name = 'IDENTITY_SERVICE_ID' and table_schema = database());
set @stmt := if( @count_columns < 1, 'ALTER TABLE IAM_HISTORY ADD IDENTITY_SERVICE_ID INT NULL AFTER ID;', 'select ''INFO: IAM_HISTORY not found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
set @stmt := if( @count_columns < 1, 'UPDATE IAM_HISTORY SET "IDENTITY_SERVICE_ID"=0;', 'select ''INFO: UPDATE''');
PREPARE statement FROM @stmt;
EXECUTE statement;
set @stmt := if( @count_columns < 1, 'ALTER TABLE IAM_HISTORY MODIFY IDENTITY_SERVICE_ID INT NOT NULL DEFAULT 0;', 'select ''INFO: NOT NULL''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;


/* IAM_IDENTITY_SERVICES OIDC */
START TRANSACTION; 
set @stmt := 'UPDATE IAM_IDENTITY_SERVICES SET "IDENTITY_SERVICE_TYPE"=''OIDC-JOC'' WHERE "IDENTITY_SERVICE_TYPE"=''OIDC'' AND ID IN (SELECT "IDENTITY_SERVICE_ID" FROM IAM_ACCOUNTS)';
PREPARE statement FROM @stmt;
EXECUTE statement;
set @stmt := 'UPDATE JOC_CONFIGURATIONS SET OBJECT_TYPE=''OIDC-JOC'' WHERE OBJECT_TYPE=''OIDC'' AND ID IN 
(SELECT T.ID FROM (SELECT ID FROM JOC_CONFIGURATIONS JC
WHERE JC.OBJECT_TYPE=''OIDC'' AND
JC.NAME IN (SELECT IC.IDENTITY_SERVICE_NAME FROM IAM_IDENTITY_SERVICES IC, IAM_ACCOUNTS IA WHERE IA.IDENTITY_SERVICE_ID=IC.ID AND  IC.IDENTITY_SERVICE_NAME=JC.NAME)) T)';
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

/* IAM_IDENTITY_SERVICES columns */
START TRANSACTION; 
set @count_columns := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'IAM_IDENTITY_SERVICES' and column_name = 'SECOND_FACTOR' and table_schema = database());
set @stmt := if( @count_columns < 1, 'ALTER TABLE IAM_IDENTITY_SERVICES ADD SECOND_FACTOR TINYINT(4) NULL AFTER AUTHENTICATION_SCHEME;', 'select ''INFO: IAM_IDENTITY_SERVICES not found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
set @stmt := if( @count_columns < 1, 'UPDATE IAM_IDENTITY_SERVICES SET "SECOND_FACTOR"=0;', 'select ''INFO: UPDATE''');
PREPARE statement FROM @stmt;
EXECUTE statement;
set @stmt := if( @count_columns < 1, 'ALTER TABLE IAM_IDENTITY_SERVICES MODIFY SECOND_FACTOR TINYINT(4) NOT NULL DEFAULT 0;', 'select ''INFO: NOT NULL''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;


/* IAM_IDENTITY_SERVICES columns */
START TRANSACTION; 
set @count_columns := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'IAM_IDENTITY_SERVICES' and column_name = 'SECOND_FACTOR_IS_ID' and table_schema = database());
set @stmt := if( @count_columns < 1, 'ALTER TABLE IAM_IDENTITY_SERVICES ADD SECOND_FACTOR_IS_ID INT NULL AFTER IDENTITY_SERVICE_NAME;', 'select ''INFO: IAM_IDENTITY_SERVICES not found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;


/* IAM_IDENTITY_SERVICES columns */
START TRANSACTION; 
set @count_columns := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'IAM_IDENTITY_SERVICES' and column_name = 'SINGLE_FACTOR_CERT' and table_schema = database());
set @stmt := if( @count_columns > 0, 'ALTER TABLE IAM_IDENTITY_SERVICES DROP COLUMN SINGLE_FACTOR_CERT;', 'select ''INFO: IAM_IDENTITY_SERVICES not found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;



START TRANSACTION; 
set @count_columns := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'IAM_IDENTITY_SERVICES' and column_name = 'SINGLE_FACTOR_PWD' and table_schema = database());
set @stmt := if( @count_columns > 0, 'ALTER TABLE IAM_IDENTITY_SERVICES DROP COLUMN SINGLE_FACTOR_PWD;', 'select ''INFO: IAM_IDENTITY_SERVICES not found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;
 

/* IAM_ROLES columns */
START TRANSACTION; 
set @count_columns := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'IAM_ROLES' and column_name = 'ORDERING' and table_schema = database());
set @stmt := if( @count_columns < 1, 'ALTER TABLE IAM_ROLES ADD ORDERING INT NULL AFTER ROLE_NAME;', 'select ''INFO: IAM_ROLES not found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;


/* IAM_ACCOUNTS columns */
START TRANSACTION; 
set @count_columns := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'IAM_ACCOUNTS' and column_name = 'EMAIL' and table_schema = database());
set @stmt := if( @count_columns < 1, 'ALTER TABLE IAM_ACCOUNTS ADD EMAIL VARCHAR(255) AFTER ACCOUNT_PASSWORD;', 'select ''INFO: IAM_ACCOUNTS not found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;


START TRANSACTION;
/* ALTER TABLE IAM_ACCOUNTS ADD CONSTRAINT UNIQUE_IAM_A_IN UNIQUE ("IDENTITY_SERVICE_ID", "ACCOUNT_NAME"); */
set @exist := (SELECT count(*) FROM information_schema.TABLE_CONSTRAINTS WHERE CONSTRAINT_NAME = 'UNIQUE_IAM_A_IN' and CONSTRAINT_TYPE = 'UNIQUE' and CONSTRAINT_SCHEMA = DATABASE());
set @sqlstatement := if( @exist < 1, 'ALTER TABLE IAM_ACCOUNTS ADD CONSTRAINT UNIQUE_IAM_A_IN UNIQUE ("IDENTITY_SERVICE_ID", "ACCOUNT_NAME");', 'select ''INFO: Constraint already exists!''');
PREPARE statement FROM @sqlstatement;
EXECUTE statement;
COMMIT;

START TRANSACTION;
/* ALTER TABLE IAM_ROLES ADD CONSTRAINT UNIQUE_IAM_R_IN UNIQUE ("IDENTITY_SERVICE_ID", "ROLE_NAME"); */
set @exist := (SELECT count(*) FROM information_schema.TABLE_CONSTRAINTS WHERE CONSTRAINT_NAME = 'UNIQUE_IAM_R_IN' and CONSTRAINT_TYPE = 'UNIQUE' and CONSTRAINT_SCHEMA = DATABASE());
set @sqlstatement := if( @exist < 1, 'ALTER TABLE IAM_ROLES ADD CONSTRAINT UNIQUE_IAM_R_IN UNIQUE ("IDENTITY_SERVICE_ID", "ROLE_NAME");', 'select ''INFO: Constraint already exists!''');
PREPARE statement FROM @sqlstatement;
EXECUTE statement;
COMMIT;

START TRANSACTION;
/* ALTER TABLE IAM_IDENTITY_SERVICES ADD CONSTRAINT UNIQUE_IAM_S_N UNIQUE ("IDENTITY_SERVICE_NAME"); */
set @exist := (SELECT count(*) FROM information_schema.TABLE_CONSTRAINTS WHERE CONSTRAINT_NAME = 'UNIQUE_IAM_S_N' and CONSTRAINT_TYPE = 'UNIQUE' and CONSTRAINT_SCHEMA = DATABASE());
set @sqlstatement := if( @exist < 1, 'ALTER TABLE IAM_IDENTITY_SERVICES ADD CONSTRAINT UNIQUE_IAM_S_N UNIQUE ("IDENTITY_SERVICE_NAME");', 'select ''INFO: Constraint already exists!''');
PREPARE statement FROM @sqlstatement;
EXECUTE statement;
COMMIT;
set SQL_MODE=ANSI_QUOTES;

/* DPL_ORDERS columns */
START TRANSACTION; 
set @count_columns := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'DPL_ORDERS' and column_name = 'START_MODE' and table_schema = database());
set @stmt := if( @count_columns < 1, 'ALTER TABLE DPL_ORDERS ADD START_MODE TINYINT(4) NULL AFTER CALENDAR_ID;', 'select ''INFO: DPL_ORDERS not found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
set @stmt := if( @count_columns < 1, 'UPDATE DPL_ORDERS SET "START_MODE"=0 where PERIOD_BEGIN is null;', 'select ''INFO: UPDATE 1''');
PREPARE statement FROM @stmt;
EXECUTE statement;
set @stmt := if( @count_columns < 1, 'UPDATE DPL_ORDERS SET "START_MODE"=1 where PERIOD_BEGIN is not null;', 'select ''INFO: UPDATE 2''');
PREPARE statement FROM @stmt;
EXECUTE statement;
set @stmt := if( @count_columns < 1, 'ALTER TABLE DPL_ORDERS MODIFY START_MODE TINYINT(4) NOT NULL DEFAULT 0;', 'select ''INFO: NOT NULL''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

START TRANSACTION; 
set @count_columns := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'DPL_ORDERS' and column_name = 'ORDER_PARAMETERISATION' and table_schema = database());
set @stmt := if( @count_columns < 1, 'ALTER TABLE DPL_ORDERS ADD ORDER_PARAMETERISATION VARCHAR(1000) NULL AFTER EXPECTED_END;', 'select ''INFO: DPL_ORDERS.ORDER_PARAMETERISATION found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

START TRANSACTION; 
set @count_columns := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'DPL_ORDERS' and column_name = 'MESSAGE' and table_schema = database());
set @stmt := if( @count_columns > 0, 'ALTER TABLE DPL_ORDERS DROP COLUMN MESSAGE;', 'select ''INFO: DPL_ORDERS.MESSAGE not found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

/* DPL_ORDERS drop indexes */
START TRANSACTION; 
set @count_indexes := (SELECT count(*) FROM information_schema.statistics WHERE table_name = 'DPL_ORDERS' and index_name = 'SOS_JS_ORDER_PLANNED_UNIQUE' and table_schema = database());
set @stmt := if( @count_indexes > 0, 'ALTER TABLE DPL_ORDERS DROP INDEX SOS_JS_ORDER_PLANNED_UNIQUE;', 'select ''INFO: Index SOS_JS_ORDER_PLANNED_UNIQUE not found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

START TRANSACTION; 
set @count_indexes := (SELECT count(*) FROM information_schema.statistics WHERE table_name = 'DPL_ORDERS' and index_name = 'IDX_DPL_O_PSTART' and table_schema = database());
set @stmt := if( @count_indexes > 0, 'ALTER TABLE DPL_ORDERS DROP INDEX IDX_DPL_O_PSTART;', 'select ''INFO: Index IDX_DPL_O_PSTART not found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

START TRANSACTION; 
set @count_indexes := (SELECT count(*) FROM information_schema.statistics WHERE table_name = 'DPL_ORDERS' and index_name = 'IDX_DPL_O_ORDER_ID' and table_schema = database());
set @stmt := if( @count_indexes > 0, 'ALTER TABLE DPL_ORDERS DROP INDEX IDX_DPL_O_ORDER_ID;', 'select ''INFO: Index IDX_DPL_O_ORDER_ID not found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT; 

START TRANSACTION; 
set @count_indexes := (SELECT count(*) FROM information_schema.statistics WHERE table_name = 'DPL_ORDERS' and index_name = 'IDX_DPL_O_START_MODE' and table_schema = database());
set @stmt := if( @count_indexes > 0, 'ALTER TABLE DPL_ORDERS DROP INDEX IDX_DPL_O_START_MODE;', 'select ''INFO: Index IDX_DPL_O_START_MODE not found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT; 

/* DPL_ORDERS add indexes */
START TRANSACTION; 
set @count_indexes := (SELECT count(*) FROM information_schema.statistics WHERE table_name = 'DPL_ORDERS' and index_name = 'IDX_DPL_O_SHID' and table_schema = database());
set @stmt := if( @count_indexes < 1, 'CREATE INDEX IDX_DPL_O_SHID ON DPL_ORDERS("SUBMISSION_HISTORY_ID");', 'select ''INFO: Index IDX_DPL_O_SHID found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT; 

START TRANSACTION; 
set @count_indexes := (SELECT count(*) FROM information_schema.statistics WHERE table_name = 'DPL_ORDERS' and index_name = 'IDX_DPL_O_ON' and table_schema = database());
set @stmt := if( @count_indexes < 1, 'CREATE INDEX IDX_DPL_O_ON ON DPL_ORDERS("ORDER_NAME");', 'select ''INFO: Index IDX_DPL_O_ON found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

START TRANSACTION; 
set @count_indexes := (SELECT count(*) FROM information_schema.statistics WHERE table_name = 'DPL_ORDERS' and index_name = 'IDX_DPL_O_OID' and table_schema = database());
set @stmt := if( @count_indexes < 1, 'CREATE INDEX IDX_DPL_O_OID ON DPL_ORDERS("ORDER_ID");', 'select ''INFO: Index IDX_DPL_O_OID found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

START TRANSACTION; 
set @count_indexes := (SELECT count(*) FROM information_schema.statistics WHERE table_name = 'DPL_ORDERS' and index_name = 'IDX_DPL_O_PSCID' and table_schema = database());
set @stmt := if( @count_indexes < 1, 'CREATE INDEX IDX_DPL_O_PSCID ON DPL_ORDERS("PLANNED_START","CONTROLLER_ID");', 'select ''INFO: Index IDX_DPL_O_PSCID found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

START TRANSACTION; 
set @count_indexes := (SELECT count(*) FROM information_schema.statistics WHERE table_name = 'DPL_ORDERS' and index_name = 'IDX_DPL_O_WN' and table_schema = database());
set @stmt := if( @count_indexes < 1, 'CREATE INDEX IDX_DPL_O_WN ON DPL_ORDERS("WORKFLOW_NAME");', 'select ''INFO: Index IDX_DPL_O_WN found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

/* DPL_HISTORY drop indexes */
START TRANSACTION; 
set @count_indexes := (SELECT count(*) FROM information_schema.statistics WHERE table_name = 'DPL_HISTORY' and index_name = 'IDX_DPL_DPDATE' and table_schema = database());
set @stmt := if( @count_indexes > 0, 'ALTER TABLE DPL_HISTORY DROP INDEX IDX_DPL_DPDATE;', 'select ''INFO: Index IDX_DPL_DPDATE not found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

START TRANSACTION; 
set @count_indexes := (SELECT count(*) FROM information_schema.statistics WHERE table_name = 'DPL_HISTORY' and index_name = 'IDX_DPL_CATEGORY' and table_schema = database());
set @stmt := if( @count_indexes > 0, 'ALTER TABLE DPL_HISTORY DROP INDEX IDX_DPL_CATEGORY;', 'select ''INFO: Index IDX_DPL_CATEGORY not found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

START TRANSACTION; 
set @count_indexes := (SELECT count(*) FROM information_schema.statistics WHERE table_name = 'DPL_HISTORY' and index_name = 'IDX_DPL_OH_ORDER_ID' and table_schema = database());
set @stmt := if( @count_indexes > 0, 'ALTER TABLE DPL_HISTORY DROP INDEX IDX_DPL_OH_ORDER_ID;', 'select ''INFO: Index IDX_DPL_OH_ORDER_ID not found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

/* DPL_HISTORY add indexes */
START TRANSACTION; 
set @count_indexes := (SELECT count(*) FROM information_schema.statistics WHERE table_name = 'DPL_HISTORY' and index_name = 'IDX_DPL_H_DPDSCID' and table_schema = database());
set @stmt := if( @count_indexes < 1, 'CREATE INDEX IDX_DPL_H_DPDSCID ON DPL_HISTORY("DAILY_PLAN_DATE","SUBMITTED","CONTROLLER_ID");', 'select ''INFO: Index IDX_DPL_H_DPDSCID found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT; 

START TRANSACTION; 
set @count_indexes := (SELECT count(*) FROM information_schema.statistics WHERE table_name = 'DPL_HISTORY' and index_name = 'IDX_DPL_H_OID' and table_schema = database());
set @stmt := if( @count_indexes < 1, 'CREATE INDEX IDX_DPL_H_OID ON DPL_HISTORY("ORDER_ID");', 'select ''INFO: Index IDX_DPL_H_OID found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT; 

/* DPL_SUBMISSIONS add indexes */
START TRANSACTION; 
set @count_indexes := (SELECT count(*) FROM information_schema.statistics WHERE table_name = 'DPL_SUBMISSIONS' and index_name = 'IDX_DPL_S_SFDCID' and table_schema = database());
set @stmt := if( @count_indexes < 1, 'CREATE INDEX IDX_DPL_S_SFDCID ON DPL_SUBMISSIONS("SUBMISSION_FOR_DATE","CONTROLLER_ID");', 'select ''INFO: Index IDX_DPL_S_SFDCID found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;
set SQL_MODE=ANSI_QUOTES;

/* DPL_ORDER_VARIABLES columns */
START TRANSACTION; 
set @count_columns := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'DPL_ORDER_VARIABLES' and column_name = 'ORDER_ID' and table_schema = database());
set @stmt := if( @count_columns < 1, 'DELETE FROM DPL_ORDER_VARIABLES WHERE "PLANNED_ORDER_ID" NOT IN (SELECT "ID" FROM DPL_ORDERS);', 'select ''INFO: ORDER_ID found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
set @stmt := if( @count_columns < 1, 'ALTER TABLE DPL_ORDER_VARIABLES ADD "ORDER_ID"  VARCHAR(255) NULL AFTER "ID";', 'select ''INFO: ORDER_ID found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
set @stmt := if( @count_columns < 1, 'ALTER TABLE DPL_ORDER_VARIABLES ADD "CONTROLLER_ID"  VARCHAR(100) NULL AFTER "ID";', 'select ''INFO: ORDER_ID found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
set @stmt := if( @count_columns < 1, 'UPDATE DPL_ORDER_VARIABLES ov, DPL_ORDERS o SET ov."CONTROLLER_ID"=o."CONTROLLER_ID",ov."ORDER_ID"=o."ORDER_ID" where ov."PLANNED_ORDER_ID"=o."ID";', 'select ''INFO: UPDATE 1''');
PREPARE statement FROM @stmt;
EXECUTE statement;
set @stmt := if( @count_columns < 1, 'ALTER TABLE DPL_ORDER_VARIABLES MODIFY "CONTROLLER_ID"  VARCHAR(100) NOT NULL;', 'select ''INFO: ORDER_ID found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
set @stmt := if( @count_columns < 1, 'ALTER TABLE DPL_ORDER_VARIABLES MODIFY "ORDER_ID"  VARCHAR(255) NOT NULL;', 'select ''INFO: ORDER_ID found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

/* DPL_ORDER_VARIABLES add indexes */
START TRANSACTION; 
set @count_indexes := (SELECT count(*) FROM information_schema.statistics WHERE table_name = 'DPL_ORDER_VARIABLES' and index_name = 'IDX_DPL_OV_OIDCID' and table_schema = database());
set @stmt := if( @count_indexes < 1, 'CREATE INDEX IDX_DPL_OV_OIDCID ON DPL_ORDER_VARIABLES("ORDER_ID","CONTROLLER_ID");', 'select ''INFO: Index IDX_DPL_OV_OIDCID found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

/* DPL_ORDER_VARIABLES drop indexes */
START TRANSACTION; 
set @count_indexes := (SELECT count(*) FROM information_schema.statistics WHERE table_name = 'DPL_ORDER_VARIABLES' and index_name = 'IDX_DPL_OV_POID' and table_schema = database());
set @stmt := if( @count_indexes > 0, 'ALTER TABLE DPL_ORDER_VARIABLES DROP INDEX IDX_DPL_OV_POID;', 'select ''INFO: Index IDX_DPL_OV_POID not found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

/* DPL_ORDER_VARIABLES drop columns */
START TRANSACTION; 
set @count_columns := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'DPL_ORDER_VARIABLES' and column_name = 'PLANNED_ORDER_ID' and table_schema = database());
set @stmt := if( @count_columns > 0, 'ALTER TABLE DPL_ORDER_VARIABLES DROP COLUMN "PLANNED_ORDER_ID";', 'select ''INFO: PLANNED_ORDER_ID removed''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;
set SQL_MODE=ANSI_QUOTES;

/* DPL_ORDER_VARIABLES columns */
START TRANSACTION; 
set @exist := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'DPL_ORDER_VARIABLES' and column_name = 'VARIABLE_VALUE' and upper(column_type) = 'LONGTEXT' and table_schema = database());
set @sqlstatement := if( @exist < 1, 'ALTER TABLE DPL_ORDER_VARIABLES MODIFY COLUMN "VARIABLE_VALUE" LONGTEXT NOT NULL;', 'select ''INFO: VARIABLE_VALUE LONGTEXT found''');
PREPARE statement FROM @sqlstatement;
EXECUTE statement;
COMMIT;
set SQL_MODE=ANSI_QUOTES;

/* HISTORY_AGENTS */
START TRANSACTION; 
set @exist := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'HISTORY_AGENTS' and column_name = 'AGENT_ID' and upper(column_type) = 'VARCHAR(100)' and table_schema = database());
set @sqlstatement := if( @exist > 0, 'ALTER TABLE HISTORY_AGENTS MODIFY COLUMN AGENT_ID VARCHAR(255) NOT NULL;', 'select ''INFO: AGENT_ID VARCHAR(100) not found''');
PREPARE statement FROM @sqlstatement;
EXECUTE statement;
COMMIT;

START TRANSACTION; 
set @exist := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'HISTORY_AGENTS' and column_name = 'URI' and upper(column_type) = 'VARCHAR(100)' and table_schema = database());
set @sqlstatement := if( @exist > 0, 'ALTER TABLE HISTORY_AGENTS MODIFY COLUMN URI VARCHAR(255) NOT NULL;', 'select ''INFO: URI VARCHAR(100) not found''');
PREPARE statement FROM @sqlstatement;
EXECUTE statement;
COMMIT;

/* HISTORY_ORDERS columns */
START TRANSACTION; 
set @count_columns := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'HISTORY_ORDERS' and column_name = 'END_MESSAGE' and table_schema = database());
set @stmt := if( @count_columns < 1, 'ALTER TABLE HISTORY_ORDERS ADD "END_MESSAGE" VARCHAR(500) NULL AFTER "END_HOS_ID";', 'select ''INFO: END_MESSAGE found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

START TRANSACTION; 
set @count_columns := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'HISTORY_ORDERS' and column_name = 'END_RETURN_CODE' and table_schema = database());
set @stmt := if( @count_columns < 1, 'ALTER TABLE HISTORY_ORDERS ADD "END_RETURN_CODE" INT(10) NULL AFTER "END_HOS_ID";', 'select ''INFO: END_RETURN_CODE found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

/* HISTORY_ORDERS indexes */
START TRANSACTION; 
set @count_indexes := (SELECT count(*) FROM information_schema.statistics WHERE table_name = 'HISTORY_ORDERS' and index_name = 'IDX_HO_LID' and table_schema = database());
set @stmt := if( @count_indexes < 1, 'CREATE INDEX IDX_HO_LID ON HISTORY_ORDERS("LOG_ID");', 'select ''INFO: Index IDX_HO_LID found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

START TRANSACTION; 
set @count_indexes := (SELECT count(*) FROM information_schema.statistics WHERE table_name = 'HISTORY_ORDERS' and index_name = 'IDX_HO_WNAME' and table_schema = database());
set @stmt := if( @count_indexes < 1, 'CREATE INDEX IDX_HO_WNAME ON HISTORY_ORDERS("WORKFLOW_NAME");', 'select ''INFO: Index IDX_HO_WNAME found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

/* HISTORY_ORDER_STEPS columns */
START TRANSACTION; 
set @exist := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'HISTORY_ORDER_STEPS' and column_name = 'AGENT_ID' and upper(column_type) = 'VARCHAR(100)' and table_schema = database());
set @sqlstatement := if( @exist > 0, 'ALTER TABLE HISTORY_ORDER_STEPS MODIFY COLUMN AGENT_ID VARCHAR(255) NOT NULL;', 'select ''INFO: AGENT_ID VARCHAR(100) not found''');
PREPARE statement FROM @sqlstatement;
EXECUTE statement;
COMMIT;

START TRANSACTION; 
set @exist := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'HISTORY_ORDER_STEPS' and column_name = 'AGENT_URI' and upper(column_type) = 'VARCHAR(100)' and table_schema = database());
set @sqlstatement := if( @exist > 0, 'ALTER TABLE HISTORY_ORDER_STEPS MODIFY COLUMN AGENT_URI VARCHAR(255) NOT NULL;', 'select ''INFO: AGENT_URI VARCHAR(100) not found''');
PREPARE statement FROM @sqlstatement;
EXECUTE statement;
COMMIT;

START TRANSACTION; 
set @count_columns := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'HISTORY_ORDER_STEPS' and column_name = 'JOB_NOTIFICATION' and table_schema = database());
set @stmt := if( @count_columns < 1, 'ALTER TABLE HISTORY_ORDER_STEPS ADD "JOB_NOTIFICATION" VARCHAR(1000) NULL AFTER "JOB_TITLE";', 'select ''INFO: JOB_NOTIFICATION found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

START TRANSACTION; 
set @count_columns := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'HISTORY_ORDER_STEPS' and column_name = 'AGENT_NAME' and table_schema = database());
set @stmt := if( @count_columns < 1, 'ALTER TABLE HISTORY_ORDER_STEPS ADD "AGENT_NAME" VARCHAR(255) NULL AFTER "AGENT_ID";', 'select ''INFO: AGENT_NAME found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

START TRANSACTION; 
set @count_columns := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'HISTORY_ORDER_STEPS' and column_name = 'SUBAGENT_CLUSTER_ID' and table_schema = database());
set @stmt := if( @count_columns < 1, 'ALTER TABLE HISTORY_ORDER_STEPS ADD "SUBAGENT_CLUSTER_ID" VARCHAR(255) NULL AFTER "AGENT_URI";', 'select ''INFO: SUBAGENT_CLUSTER_ID found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

/* HISTORY_ORDER_STEPS indexes */
START TRANSACTION; 
set @count_indexes := (SELECT count(*) FROM information_schema.statistics WHERE table_name = 'HISTORY_ORDER_STEPS' and index_name = 'IDX_HOSTEPS_WNAME' and table_schema = database());
set @stmt := if( @count_indexes < 1, 'CREATE INDEX IDX_HOSTEPS_WNAME ON HISTORY_ORDER_STEPS("WORKFLOW_NAME");', 'select ''INFO: Index IDX_HOSTEPS_WNAME found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

/* HISTORY_LOGS indexes */
START TRANSACTION; 
set @count_indexes := (SELECT count(*) FROM information_schema.statistics WHERE table_name = 'HISTORY_LOGS' and index_name = 'IDX_HLOGS_HOMPID' and table_schema = database());
set @stmt := if( @count_indexes < 1, 'CREATE INDEX IDX_HLOGS_HOMPID ON HISTORY_LOGS("HO_MAIN_PARENT_ID");', 'select ''INFO: Index IDX_HLOGS_HOMPID found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

/* add HISTORY_ORDER_TAGS.GROUP_ID */
START TRANSACTION;
set @count_columns := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'HISTORY_ORDER_TAGS' and column_name = 'GROUP_ID' and table_schema = database());
set @stmt := if( @count_columns < 1, 'ALTER TABLE HISTORY_ORDER_TAGS ADD "GROUP_ID" INT UNSIGNED NOT NULL DEFAULT 0 AFTER "ORDERING";', 'select ''INFO: GROUP_ID found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

/* DROP tables */
DROP TABLE IF EXISTS HISTORY_TEMP_LOGS;
 
set SQL_MODE=ANSI_QUOTES;

/* HISTORY_ORDERS columns */
START TRANSACTION; 
set @exist := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'HISTORY_ORDERS' and column_name = 'START_VARIABLES' and upper(column_type) = 'LONGTEXT' and table_schema = database());
set @sqlstatement := if( @exist < 1, 'ALTER TABLE HISTORY_ORDERS MODIFY COLUMN "START_VARIABLES" LONGTEXT NULL;', 'select ''INFO: START_VARIABLES LONGTEXT found''');
PREPARE statement FROM @sqlstatement;
EXECUTE statement;
COMMIT;
set SQL_MODE=ANSI_QUOTES;

/* INV_AGENT_INSTANCES columns */
START TRANSACTION; 
set @count_columns := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'INV_AGENT_INSTANCES' and column_name = 'JAVA_VERSION' and table_schema = database());
set @stmt := if( @count_columns < 1, 'ALTER TABLE INV_AGENT_INSTANCES ADD "JAVA_VERSION" VARCHAR(30) NULL AFTER "VERSION";', 'select ''INFO: JAVA_VERSION found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

START TRANSACTION; 
set @count_columns := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'INV_AGENT_INSTANCES' and column_name = 'TITLE' and table_schema = database());
set @stmt := if( @count_columns < 1, 'ALTER TABLE INV_AGENT_INSTANCES ADD "TITLE" VARCHAR(255) NULL DEFAULT 0 AFTER "DISABLED";', 'select ''INFO: TITLE found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

START TRANSACTION; 
set @count_columns := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'INV_AGENT_INSTANCES' and column_name = 'DEPLOYED' and table_schema = database());
set @stmt := if( @count_columns < 1, 'ALTER TABLE INV_AGENT_INSTANCES ADD "DEPLOYED" TINYINT NOT NULL DEFAULT 0 AFTER "DISABLED";', 'select ''INFO: DEPLOYED found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

START TRANSACTION; 
set @count_columns := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'INV_AGENT_INSTANCES' and column_name = 'HIDDEN' and table_schema = database());
set @stmt := if( @count_columns < 1, 'ALTER TABLE INV_AGENT_INSTANCES CHANGE COLUMN "DISABLED" "HIDDEN" TINYINT NOT NULL DEFAULT 0 AFTER "URI";', 'select ''INFO: HIDDEN found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

START TRANSACTION; 
set @count_columns := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'INV_AGENT_INSTANCES' and column_name = 'DISABLED' and table_schema = database());
set @stmt := if( @count_columns < 1, 'ALTER TABLE INV_AGENT_INSTANCES ADD "DISABLED" TINYINT NOT NULL DEFAULT 0 AFTER "URI";', 'select ''INFO: DISABLED found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

START TRANSACTION; 
set @count_columns := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'INV_AGENT_INSTANCES' and column_name = 'ORDERING' and table_schema = database());
set @stmt := if( @count_columns < 1, 'ALTER TABLE INV_AGENT_INSTANCES ADD "ORDERING" INT NOT NULL DEFAULT 0 AFTER "URI";', 'select ''INFO: ORDERING found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

START TRANSACTION; 
set @count_columns := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'INV_AGENT_INSTANCES' and column_name = 'PROCESS_LIMIT' and table_schema = database());
set @stmt := if( @count_columns < 1, 'ALTER TABLE INV_AGENT_INSTANCES ADD "PROCESS_LIMIT" INT NULL AFTER "URI";', 'select ''INFO: PROCESS_LIMIT found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

/* INV_JS_INSTANCES columns */
START TRANSACTION; 
set @count_columns := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'INV_JS_INSTANCES' and column_name = 'JAVA_VERSION' and table_schema = database());
set @stmt := if( @count_columns < 1, 'ALTER TABLE INV_JS_INSTANCES ADD "JAVA_VERSION" VARCHAR(30) NULL AFTER "VERSION";', 'select ''INFO: JAVA_VERSION found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;


/* only temp. for delevopers INV_JS_INSTANCES columns */
START TRANSACTION; 
set @count_columns := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'INV_SUBAGENT_INSTANCES' and column_name = 'ORDERING' and table_schema = database());
set @stmt := if( @count_columns < 1, 'ALTER TABLE INV_SUBAGENT_INSTANCES ADD "ORDERING" INT NOT NULL DEFAULT 0 AFTER "IS_DIRECTOR";', 'select ''INFO: ORDERING found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;


/* INV_SUBAGENT_INSTANCES */
START TRANSACTION; 
set @count_columns := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'INV_SUBAGENT_INSTANCES' and column_name = 'TITLE' and table_schema = database());
set @stmt := if( @count_columns < 1, 'ALTER TABLE INV_SUBAGENT_INSTANCES ADD "TITLE" VARCHAR(255) NULL DEFAULT 0 AFTER "IS_DIRECTOR";', 'select ''INFO: DISABLED found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

START TRANSACTION; 
set @count_columns := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'INV_SUBAGENT_INSTANCES' and column_name = 'DEPLOYED' and table_schema = database());
set @stmt := if( @count_columns < 1, 'ALTER TABLE INV_SUBAGENT_INSTANCES ADD "DEPLOYED" TINYINT NOT NULL DEFAULT 0 AFTER "IS_DIRECTOR";', 'select ''INFO: DISABLED found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

START TRANSACTION; 
set @count_columns := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'INV_SUBAGENT_INSTANCES' and column_name = 'DISABLED' and table_schema = database());
set @stmt := if( @count_columns < 1, 'ALTER TABLE INV_SUBAGENT_INSTANCES ADD "DISABLED" TINYINT NOT NULL DEFAULT 0 AFTER "IS_DIRECTOR";', 'select ''INFO: DISABLED found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

START TRANSACTION; 
set @count_columns := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'INV_SUBAGENT_CLUSTERS' and column_name = 'ORDERING' and table_schema = database());
set @stmt := if( @count_columns < 1, 'ALTER TABLE INV_SUBAGENT_CLUSTERS ADD "ORDERING" INT NOT NULL DEFAULT 0 AFTER "DEPLOYED";', 'select ''INFO: ORDERING found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;


/* INV_CONFIGURATIONS columns */
START TRANSACTION; 
set @count_columns := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'INV_CONFIGURATIONS' and column_name = 'REPO_CTRL' and table_schema = database());
set @stmt := if( @count_columns < 1, 'ALTER TABLE INV_CONFIGURATIONS ADD "REPO_CTRL" TINYINT NOT NULL DEFAULT 0 AFTER "RELEASED";', 'select ''INFO: DISABLED found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

/* INV_CERTS */
/* DROP CONSTRAINT UNIQUE_ICS_KTC */
START TRANSACTION; 
set @exist := (SELECT count(*) FROM information_schema.TABLE_CONSTRAINTS WHERE CONSTRAINT_NAME = 'UNIQUE_ICS_KTC' and CONSTRAINT_TYPE = 'UNIQUE' and CONSTRAINT_SCHEMA = DATABASE());
set @sqlstatement := if( @exist = 1, 'ALTER TABLE INV_CERTS DROP INDEX UNIQUE_ICS_KTC;', 'select ''INFO: Constraint not exists!''');
PREPARE statement FROM @sqlstatement;
EXECUTE statement;
COMMIT;

/* add CONSTRAINT UNIQUE_ICS_KTCAS */
START TRANSACTION; 
set @exist := (SELECT count(*) FROM information_schema.TABLE_CONSTRAINTS WHERE CONSTRAINT_NAME = 'UNIQUE_ICS_KTCAS' and CONSTRAINT_TYPE = 'UNIQUE' and CONSTRAINT_SCHEMA = DATABASE());
set @sqlstatement := if( @exist < 1, 'ALTER TABLE INV_CERTS ADD CONSTRAINT UNIQUE_ICS_KTCAS UNIQUE ("KEY_TYPE", "CA", "ACCOUNT", "SECLVL");', 'select ''INFO: Constraint already exists!''');
PREPARE statement FROM @sqlstatement;
EXECUTE statement;
COMMIT;


/* INV_FAVORITES columns */
START TRANSACTION; 
set @count_columns := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'INV_FAVORITES' and column_name = 'FAVORIT' and table_schema = database());
set @stmt := if( @count_columns = 1, 'ALTER TABLE INV_FAVORITES CHANGE COLUMN "FAVORIT" "FAVORITE" LONGTEXT NULL;', 'select ''INFO: FAVORIT not found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

/* drop INV_FAVORITES CONSTRAINT UNIQUE_IF_TN */
START TRANSACTION; 
set @exist := (SELECT count(*) FROM information_schema.TABLE_CONSTRAINTS WHERE CONSTRAINT_NAME = 'UNIQUE_IF_TN' and CONSTRAINT_TYPE = 'UNIQUE' and CONSTRAINT_SCHEMA = DATABASE());
set @sqlstatement := if( @exist = 1, 'ALTER TABLE INV_FAVORITES DROP INDEX UNIQUE_IF_TN;', 'select ''INFO: Constraint not exists!''');
PREPARE statement FROM @sqlstatement;
EXECUTE statement;
COMMIT;

/* add INV_FAVORITES CONSTRAINT UNIQUE_IF_TNA */
START TRANSACTION; 
set @exist := (SELECT count(*) FROM information_schema.TABLE_CONSTRAINTS WHERE CONSTRAINT_NAME = 'UNIQUE_IF_TNA' and CONSTRAINT_TYPE = 'UNIQUE' and CONSTRAINT_SCHEMA = DATABASE());
set @sqlstatement := if( @exist < 1, 'ALTER TABLE INV_FAVORITES ADD CONSTRAINT UNIQUE_IF_TNA UNIQUE ("TYPE","NAME","ACCOUNT");', 'select ''INFO: Constraint already exists!''');
PREPARE statement FROM @sqlstatement;
EXECUTE statement;
COMMIT;

/* alter INV_SUBAGENT_CLUSTER_MEMBERS.PRIORITY */
START TRANSACTION;
set @exist := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'INV_SUBAGENT_CLUSTER_MEMBERS' and column_name = 'PRIORITY' and upper(data_type) = 'INT' and table_schema = database());
set @stmt := if( @exist = 1, 'ALTER TABLE INV_SUBAGENT_CLUSTER_MEMBERS MODIFY "PRIORITY" VARCHAR(255) NOT NULL;', 'select ''INFO: PRIORITY not found or is already of type varchar''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

/* alter INV_TAGS.GROUP_ID */
START TRANSACTION;
set @count_columns := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'INV_TAGS' and column_name = 'GROUP_ID' and table_schema = database());
set @stmt := if( @count_columns < 1, 'ALTER TABLE INV_TAGS ADD "GROUP_ID" INT UNSIGNED NOT NULL DEFAULT 0 AFTER "ORDERING";', 'select ''INFO: GROUP_ID found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

/* alter INV_JOB_TAGS.GROUP_ID */
START TRANSACTION;
set @count_columns := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'INV_JOB_TAGS' and column_name = 'GROUP_ID' and table_schema = database());
set @stmt := if( @count_columns < 1, 'ALTER TABLE INV_JOB_TAGS ADD "GROUP_ID" INT UNSIGNED NOT NULL DEFAULT 0 AFTER "ORDERING";', 'select ''INFO: GROUP_ID found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;
set SQL_MODE=ANSI_QUOTES;

/* JOC_INSTANCES columns */
START TRANSACTION; 
set @count_columns := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'JOC_INSTANCES' and column_name = 'CLUSTER_ID' and table_schema = database());
set @stmt := if( @count_columns < 1, 'ALTER TABLE JOC_INSTANCES ADD "CLUSTER_ID" VARCHAR(10) NULL AFTER "ID";', 'select ''INFO: CLUSTER_ID found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
set @stmt := if( @count_columns < 1, 'UPDATE JOC_INSTANCES SET "CLUSTER_ID"=''${jocClusterId}'';', 'select ''INFO: UPDATE joc''');
PREPARE statement FROM @stmt;
EXECUTE statement;
set @stmt := if( @count_columns < 1, 'ALTER TABLE JOC_INSTANCES MODIFY "CLUSTER_ID" VARCHAR(10) NOT NULL;', 'select ''INFO: NOT NULL''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

START TRANSACTION; 
set @count_columns := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'JOC_INSTANCES' and column_name = 'CERTIFICATE' and table_schema = database());
set @stmt := if( @count_columns < 1, 'ALTER TABLE JOC_INSTANCES ADD "CERTIFICATE" VARCHAR(4000) NULL;', 'select ''INFO: CERTIFICATE found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
set @count_columns := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'JOC_INSTANCES' and column_name = 'API_SERVER' and table_schema = database());
set @stmt := if( @count_columns < 1, 'ALTER TABLE JOC_INSTANCES ADD "API_SERVER" TINYINT(1) UNSIGNED NULL;', 'select ''INFO: API_SERVER found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
set @stmt := if( @count_columns < 1, 'UPDATE JOC_INSTANCES SET "API_SERVER"=0;', 'select ''INFO: UPDATE joc''');
PREPARE statement FROM @stmt;
EXECUTE statement;
set @stmt := if( @count_columns < 1, 'ALTER TABLE JOC_INSTANCES MODIFY "API_SERVER" TINYINT(1) UNSIGNED NOT NULL;', 'select ''INFO: NOT NULL''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

START TRANSACTION; 
set @count_columns := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'JOC_INSTANCES' and column_name = 'VERSION' and table_schema = database());
set @stmt := if( @count_columns < 1, 'ALTER TABLE JOC_INSTANCES ADD "VERSION" VARCHAR(30) NULL;', 'select ''INFO: VERSION found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;
set SQL_MODE=ANSI_QUOTES;

/* MON_NOTIFICATIONS columns */
START TRANSACTION; 
set @count_columns := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'MON_NOTIFICATIONS' and column_name = 'WARN' and table_schema = database());
set @stmt := if( @count_columns < 1, 'ALTER TABLE MON_NOTIFICATIONS ADD "WARN" TINYINT UNSIGNED NULL AFTER "HAS_MONITORS";', 'select ''INFO: WARN found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
set @stmt := if( @count_columns < 1, 'UPDATE MON_NOTIFICATIONS SET "WARN"=0;', 'select ''INFO: WARN found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
set @stmt := if( @count_columns < 1, 'ALTER TABLE MON_NOTIFICATIONS MODIFY "WARN" TINYINT UNSIGNED NOT NULL;', 'select ''INFO: WARN found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

START TRANSACTION; 
set @count_columns := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'MON_NOTIFICATIONS' and column_name = 'WARN_TEXT' and table_schema = database());
set @stmt := if( @count_columns < 1, 'ALTER TABLE MON_NOTIFICATIONS ADD "WARN_TEXT" VARCHAR(500) NULL AFTER "WARN";', 'select ''INFO: WARN_TEXT found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

/* MON_ORDERS columns */
START TRANSACTION; 
set @count_columns := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'MON_ORDERS' and column_name = 'END_MESSAGE' and table_schema = database());
set @stmt := if( @count_columns < 1, 'ALTER TABLE MON_ORDERS ADD "END_MESSAGE" VARCHAR(500) NULL AFTER "END_HOS_ID";', 'select ''INFO: END_MESSAGE found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

START TRANSACTION; 
set @count_columns := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'MON_ORDERS' and column_name = 'END_RETURN_CODE' and table_schema = database());
set @stmt := if( @count_columns < 1, 'ALTER TABLE MON_ORDERS ADD "END_RETURN_CODE" INT(10) NULL AFTER "END_HOS_ID";', 'select ''INFO: END_RETURN_CODE found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

/* MON_ORDERS indexes */
START TRANSACTION; 
set @count_indexes := (SELECT count(*) FROM information_schema.statistics WHERE table_name = 'MON_ORDERS' and index_name = 'IDX_MONO_MPID' and table_schema = database());
set @stmt := if( @count_indexes < 1, 'CREATE INDEX IDX_MONO_MPID ON MON_ORDERS("MAIN_PARENT_ID");', 'select ''INFO: Index IDX_MONO_MPID found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

/* MON_ORDER_STEPS drop index */
START TRANSACTION; 
set @count_indexes := (SELECT count(*) FROM information_schema.statistics WHERE table_name = 'MON_ORDER_STEPS' and index_name = 'IDX_MONOS_W' and table_schema = database());
set @stmt := if( @count_indexes > 0, 'ALTER TABLE MON_ORDER_STEPS DROP INDEX IDX_MONOS_W;', 'select ''INFO: Index IDX_MONOS_W not found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

/* MON_ORDER_STEPS columns */
START TRANSACTION; 
set @exist := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'MON_ORDER_STEPS' and column_name = 'AGENT_ID' and upper(column_type) = 'VARCHAR(100)' and table_schema = database());
set @sqlstatement := if( @exist > 0, 'ALTER TABLE MON_ORDER_STEPS MODIFY COLUMN AGENT_ID VARCHAR(255) NOT NULL;', 'select ''INFO: AGENT_ID VARCHAR(100) not found''');
PREPARE statement FROM @sqlstatement;
EXECUTE statement;
COMMIT;

START TRANSACTION; 
set @exist := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'MON_ORDER_STEPS' and column_name = 'AGENT_URI' and upper(column_type) = 'VARCHAR(100)' and table_schema = database());
set @sqlstatement := if( @exist > 0, 'ALTER TABLE MON_ORDER_STEPS MODIFY COLUMN AGENT_URI VARCHAR(255) NOT NULL;', 'select ''INFO: AGENT_URI VARCHAR(100) not found''');
PREPARE statement FROM @sqlstatement;
EXECUTE statement;
COMMIT;

START TRANSACTION; 
set @count_columns := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'MON_ORDER_STEPS' and column_name = 'JOB_NOTIFICATION' and table_schema = database());
set @stmt := if( @count_columns < 1, 'ALTER TABLE MON_ORDER_STEPS ADD "JOB_NOTIFICATION" VARCHAR(1000) NULL AFTER "JOB_TITLE";', 'select ''INFO: JOB_NOTIFICATION found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

START TRANSACTION; 
set @count_columns := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'MON_ORDER_STEPS' and column_name = 'AGENT_NAME' and table_schema = database());
set @stmt := if( @count_columns < 1, 'ALTER TABLE MON_ORDER_STEPS ADD "AGENT_NAME" VARCHAR(255) NULL AFTER "AGENT_ID";', 'select ''INFO: AGENT_NAME found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

START TRANSACTION; 
set @count_columns := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'MON_ORDER_STEPS' and column_name = 'SUBAGENT_CLUSTER_ID' and table_schema = database());
set @stmt := if( @count_columns < 1, 'ALTER TABLE MON_ORDER_STEPS ADD "SUBAGENT_CLUSTER_ID" VARCHAR(255) NULL AFTER "AGENT_URI";', 'select ''INFO: SUBAGENT_CLUSTER_ID found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

START TRANSACTION; 
set @count_columns := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'MON_ORDER_STEPS' and column_name = 'WARN' and table_schema = database());
set @stmt := if( @count_columns = 1, 'ALTER TABLE MON_ORDER_STEPS DROP COLUMN WARN;', 'select ''INFO: WARN not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

START TRANSACTION; 
set @count_columns := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'MON_ORDER_STEPS' and column_name = 'WARN_TEXT' and table_schema = database());
set @stmt := if( @count_columns = 1, 'ALTER TABLE MON_ORDER_STEPS DROP COLUMN WARN_TEXT;', 'select ''INFO: WARN_TEXT not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

/* MON_ORDER_STEPS indexes */
START TRANSACTION; 
set @count_indexes := (SELECT count(*) FROM information_schema.statistics WHERE table_name = 'MON_ORDER_STEPS' and index_name = 'IDX_MONOS_HMPID' and table_schema = database());
set @stmt := if( @count_indexes < 1, 'CREATE INDEX IDX_MONOS_HMPID ON MON_ORDER_STEPS("HO_MAIN_PARENT_ID");', 'select ''INFO: Index IDX_MONOS_HMPID found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;
set SQL_MODE=ANSI_QUOTES;

/* starting with JS7 release 2.5.2 (JOC-1447) - MON_SYSNOTIFICATIONS related changes */

/* MON_NOT_MONITORS columns */
START TRANSACTION; 
set @count_columns := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'MON_NOT_MONITORS' and column_name = 'APPLICATION' and table_schema = database());
set @stmt := if( @count_columns < 1, 'ALTER TABLE MON_NOT_MONITORS ADD "APPLICATION" TINYINT UNSIGNED NULL AFTER "NOT_ID";', 'select ''INFO: APPLICATION found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
set @stmt := if( @count_columns < 1, 'UPDATE MON_NOT_MONITORS SET "APPLICATION"=0;', 'select ''INFO: APPLICATION found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
set @stmt := if( @count_columns < 1, 'ALTER TABLE MON_NOT_MONITORS MODIFY "APPLICATION" TINYINT UNSIGNED NOT NULL;', 'select ''INFO: APPLICATION found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

/* MON_NOT_MONITORS drop index */
START TRANSACTION; 
set @count_indexes := (SELECT count(*) FROM information_schema.statistics WHERE table_name = 'MON_NOT_MONITORS' and index_name = 'IDX_MONM_NID' and table_schema = database());
set @stmt := if( @count_indexes > 0, 'ALTER TABLE MON_NOT_MONITORS DROP INDEX IDX_MONM_NID;', 'select ''INFO: Index IDX_MONM_NID not found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

/* MON_NOT_MONITORS create index */
START TRANSACTION; 
set @count_indexes := (SELECT count(*) FROM information_schema.statistics WHERE table_name = 'MON_NOT_MONITORS' and index_name = 'IDX_MONM_NIDA' and table_schema = database());
set @stmt := if( @count_indexes < 1, 'CREATE INDEX IDX_MONM_NIDA ON MON_NOT_MONITORS("NOT_ID","APPLICATION");', 'select ''INFO: Index IDX_MONM_NIDA found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

/* MON_NOT_ACKNOWLEDGEMENTS columns */
START TRANSACTION; 
set @count_columns := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'MON_NOT_ACKNOWLEDGEMENTS' and column_name = 'APPLICATION' and table_schema = database());
set @stmt := if( @count_columns < 1, 'ALTER TABLE MON_NOT_ACKNOWLEDGEMENTS ADD "APPLICATION" TINYINT UNSIGNED NULL AFTER "NOT_ID";', 'select ''INFO: APPLICATION found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
set @stmt := if( @count_columns < 1, 'UPDATE MON_NOT_ACKNOWLEDGEMENTS SET "APPLICATION"=0;', 'select ''INFO: APPLICATION found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
set @stmt := if( @count_columns < 1, 'ALTER TABLE MON_NOT_ACKNOWLEDGEMENTS MODIFY "APPLICATION" TINYINT UNSIGNED NOT NULL;', 'select ''INFO: APPLICATION found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

/* MON_NOT_ACKNOWLEDGEMENTS primary key */
START TRANSACTION; 
set @count_indexes := (SELECT count(*) FROM information_schema.statistics WHERE table_name = 'MON_NOT_ACKNOWLEDGEMENTS' and upper(index_name) = 'PRIMARY' and table_schema = database());
set @stmt := if( @count_indexes < 2, 'ALTER TABLE MON_NOT_ACKNOWLEDGEMENTS DROP PRIMARY KEY, ADD PRIMARY KEY ("NOT_ID","APPLICATION");', 'select ''INFO: Compound index PRIMARY found''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;



set SQL_MODE=ANSI_QUOTES;

/* MON_ORDERS columns */
START TRANSACTION; 
set @exist := (SELECT count(*) FROM information_schema.columns WHERE table_name = 'MON_ORDERS' and column_name = 'START_VARIABLES' and upper(column_type) = 'LONGTEXT' and table_schema = database());
set @sqlstatement := if( @exist < 1, 'ALTER TABLE MON_ORDERS MODIFY COLUMN "START_VARIABLES" LONGTEXT NULL;', 'select ''INFO: START_VARIABLES LONGTEXT found''');
PREPARE statement FROM @sqlstatement;
EXECUTE statement;
COMMIT;
/*
 | Insert Script for default authentication account root:root for identity services
 | SOS GmbH, 2021-12-12
*/
set SQL_MODE=ANSI_QUOTES;

START TRANSACTION; 

set @row_exists1 := (SELECT EXISTS(SELECT * FROM IAM_IDENTITY_SERVICES));
set @row_exists2 := (SELECT EXISTS(SELECT * FROM IAM_ACCOUNT2ROLES));
set @row_exists3 := (SELECT EXISTS(SELECT * FROM IAM_ACCOUNTS));
set @row_exists4 := (SELECT EXISTS(SELECT * FROM IAM_PERMISSIONS));
set @row_exists5 := (SELECT EXISTS(SELECT * FROM IAM_ROLES));


set @sql1 := 'INSERT INTO IAM_IDENTITY_SERVICES(IDENTITY_SERVICE_TYPE, "IDENTITY_SERVICE_NAME","REQUIRED","DISABLED","AUTHENTICATION_SCHEME", "SECOND_FACTOR", "ORDERING") VALUES (''JOC'',''JOC-INITIAL'',0,0,''SINGLE'',0,0);';
set @sql2 := 'INSERT INTO IAM_ACCOUNTS("IDENTITY_SERVICE_ID", "ACCOUNT_NAME","ACCOUNT_PASSWORD") VALUES (@LAST_ID_IN_IAM_IDENTITY_SERVICES,''root'',''$JS7-1.0$65536$9aRojy9RBlLyf40BVcE+pg==$QLXLz0CHFaYDNeIE3ioZIOAGoBy5xo2rTmp7i38DEEF5cK22MzpAL89jA2USWv5KfTth8yprRkvIk+iWS+q3Aw=='');';
set @sql3 := 'INSERT INTO IAM_ROLES("IDENTITY_SERVICE_ID", "ROLE_NAME") VALUES (@LAST_ID_IN_IAM_IDENTITY_SERVICES,''all'');';
set @sql4 := 'INSERT INTO IAM_ACCOUNT2ROLES("ROLE_ID", "ACCOUNT_ID") VALUES (@LAST_ID_IN_IAM_ROLES,@LAST_ID_IN_IAM_ACCOUNTS);';
set @sql5 := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products'',0,0);';

/*
 Administrator
*/
set @sqlr1   := 'INSERT INTO IAM_ROLES("IDENTITY_SERVICE_ID", "ROLE_NAME") VALUES (@LAST_ID_IN_IAM_IDENTITY_SERVICES,''administrator'');';
set @sqlp11 := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:joc:administration'',0,0);';
set @sqlp12 := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:joc:auditlog:view'',0,0);';
set @sqlp13 := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:joc:calendars:view'',0,0);';
set @sqlp14 := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:joc:cluster:manage'',0,0);';
set @sqlp15 := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:joc:dailyplan:view'',0,0);';
set @sqlp16 := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:joc:documentations:view'',0,0);';
set @sqlp17 := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:joc:inventory:view'',0,0);';
set @sqlp18 := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:joc:notification'',0,0);';
set @sqlp19 := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:joc:others'',0,0);';
set @sqlp110 := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:controller:view'',0,0);';
set @sqlp111 := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:controller:restart'',0,0);';
set @sqlp112 := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:controller:terminate'',0,0);';
set @sqlp113 := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:controller:switch_over'',0,0);';
set @sqlp114 := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:controller:agents:view'',0,0);';
set @sqlp115 := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:controller:deployment:view'',0,0);';
set @sqlp116 := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:controller:locks:view'',0,0);';
set @sqlp117 := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:controller:workflows:view'',0,0);';
set @sqlp118 := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:controller:orders:view'',0,0);';

/*
 application_manager
*/
set @sqlr2   := 'INSERT INTO IAM_ROLES("IDENTITY_SERVICE_ID", "ROLE_NAME") VALUES (@LAST_ID_IN_IAM_IDENTITY_SERVICES,''application_manager'');';
set @sqlp21  := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:joc:administration:controller:view'',0,0);';
set @sqlp22  := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:joc:administration:customization'',0,0);';
set @sqlp23  := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:joc:administration:settings'',0,0);';
set @sqlp24  := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:joc:auditlog:view'',0,0);';
set @sqlp25  := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:joc:calendars:view'',0,0);';
set @sqlp26  := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:joc:dailyplan'',0,0);';
set @sqlp27  := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:joc:documentations'',0,0);';
set @sqlp28  := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:joc:filetransfer'',0,0);';
set @sqlp29  := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:joc:inventory'',0,0);';
set @sqlp210 := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:joc:notification:view'',0,0);';
set @sqlp211 := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:controller:view'',0,0);';
set @sqlp212 := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:controller:agents:view'',0,0);';
set @sqlp213 := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:controller:deployment'',0,0);';
set @sqlp214 := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:controller:locks:view'',0,0);';
set @sqlp215 := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:controller:workflows:view'',0,0);';
set @sqlp216 := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:controller:orders'',0,0);';
 
/*
 it_operator
*/
 
set @sqlr3    := 'INSERT INTO IAM_ROLES("IDENTITY_SERVICE_ID", "ROLE_NAME") VALUES (@LAST_ID_IN_IAM_IDENTITY_SERVICES,''it_operator'');';
set @sqlp31   := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:joc:administration:customization'',0,0);';
set @sqlp32   := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:joc:administration:settings:view'',0,0);';
set @sqlp33   := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:joc:auditlog:view'',0,0);';
set @sqlp34   := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:joc:calendars:view'',0,0);';
set @sqlp35   := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:joc:dailyplan'',0,0);';
set @sqlp36   := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:joc:documentations'',0,0);';
set @sqlp37   := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:joc:filetransfer'',0,0);';
set @sqlp38   := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:joc:inventory:view'',0,0);';
set @sqlp39   := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:joc:notification:view'',0,0);';
set @sqlp310  := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:controller:view'',0,0);';
set @sqlp311  := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:controller:agents:view'',0,0);';
set @sqlp312  := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:controller:deployment:view'',0,0);';
set @sqlp313  := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:controller:locks:view'',0,0);';
set @sqlp314  := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:controller:workflows:view'',0,0);';
set @sqlp315  := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:controller:orders'',0,0);';


/*
 incident_manager
*/
 
set @sqlr4    := 'INSERT INTO IAM_ROLES("IDENTITY_SERVICE_ID", "ROLE_NAME") VALUES (@LAST_ID_IN_IAM_IDENTITY_SERVICES,''incident_manager'');';
set @sqlp41   := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:joc:administration:accounts'',0,0);';
set @sqlp42   := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:joc:administration:customization'',0,0);';
set @sqlp43   := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:joc:administration:settings'',0,0);';
set @sqlp44   := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:joc:auditlog:view'',0,0);';
set @sqlp45   := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:joc:calendars:view'',0,0);';
set @sqlp46   := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:joc:dailyplan'',0,0);';
set @sqlp47   := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:joc:documentations'',0,0);';
set @sqlp48   := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:joc:filetransfer'',0,0);';
set @sqlp49   := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:joc:inventory'',0,0);';
set @sqlp410  := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:joc:notification'',0,0);';
set @sqlp411  := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:joc:others'',0,0);';
set @sqlp412  := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:controller'',0,0);';



/*
 business_user
*/

set @sqlr5    := 'INSERT INTO IAM_ROLES("IDENTITY_SERVICE_ID", "ROLE_NAME") VALUES (@LAST_ID_IN_IAM_IDENTITY_SERVICES,''business_user'');';
set @sqlp51   := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:joc:administration:customization:view'',0,0);';
set @sqlp52   := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:joc:auditlog:view'',0,0);';
set @sqlp53   := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:joc:calendars:view'',0,0);';
set @sqlp54   := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:joc:dailyplan:view'',0,0);';
set @sqlp55   := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:joc:documentations:view'',0,0);';
set @sqlp56   := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:joc:filetransfer:view'',0,0);';
set @sqlp57   := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:controller:view'',0,0);';
set @sqlp58   := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:controller:agents:view'',0,0);';
set @sqlp59   := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:controller:locks:view'',0,0);';
set @sqlp510  := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:controller:workflows:view'',0,0);';
set @sqlp511  := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:controller:orders:view'',0,0);';



/*
 api_user
*/

set @sqlr6    := 'INSERT INTO IAM_ROLES("IDENTITY_SERVICE_ID", "ROLE_NAME") VALUES (@LAST_ID_IN_IAM_IDENTITY_SERVICES,''api_user'');';
set @sqlp61   := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:joc:auditlog:view'',0,0);';
set @sqlp62   := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:joc:calendars:view'',0,0);';
set @sqlp63   := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:joc:dailyplan:view'',0,0);';
set @sqlp64   := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:joc:documentations:view'',0,0);';
set @sqlp65   := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:joc:filetransfer:view'',0,0);';
set @sqlp66   := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:controller:view'',0,0);';
set @sqlp67   := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:controller:agents:view'',0,0);';
set @sqlp68   := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:controller:deployment:view'',0,0);';
set @sqlp69   := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:controller:locks:view'',0,0);';
set @sqlp610  := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:controller:workflows:view'',0,0);';
set @sqlp611  := 'INSERT INTO IAM_PERMISSIONS("IDENTITY_SERVICE_ID","ROLE_ID","ACCOUNT_PERMISSION", "EXCLUDED","RECURSIVE") VALUE (@LAST_ID_IN_IAM_IDENTITY_SERVICES,@LAST_ID_IN_IAM_ROLES,''sos:products:controller:orders:view'',0,0);';
 

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sql1, 'select ''INFO: Table IAM_IDENTITY_SERVICES not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;
SET @LAST_ID_IN_IAM_IDENTITY_SERVICES = LAST_INSERT_ID();

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sql2, 'select ''INFO: Table IAM_ACCOUNT2ROLES not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;
SET @LAST_ID_IN_IAM_ACCOUNTS = LAST_INSERT_ID();

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sql3, 'select ''INFO: Table IAM_ACCOUNTS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;
SET @LAST_ID_IN_IAM_ROLES = LAST_INSERT_ID();

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sql4, 'select ''INFO: Table IAM_ROLES not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sql5, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;


/*
 Administrator
*/

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlr1, 'select ''INFO: Table IAM_ROLES not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;
SET @LAST_ID_IN_IAM_ROLES = LAST_INSERT_ID();

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp11, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp12, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp13, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp14, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp15, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp16, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp17, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp18, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp19, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp110, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp111, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp112, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp113, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp114, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp115, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp116, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp117, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp118, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

/*
 application_manager
*/

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlr2, 'select ''INFO: Table IAM_ROLES not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;
SET @LAST_ID_IN_IAM_ROLES = LAST_INSERT_ID();

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp21, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp22, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp23, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp24, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp25, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp26, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp27, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp28, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp29, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp210, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp211, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp212, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp213, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp214, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp215, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp216, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;



/*
 it_operator
*/

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlr3, 'select ''INFO: Table IAM_ROLES not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;
SET @LAST_ID_IN_IAM_ROLES = LAST_INSERT_ID();

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp31, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp32, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp33, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp34, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp35, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp36, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp37, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp38, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp39, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp310, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp311, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp312, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp313, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp314, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp315, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;


/*
 incident_manager
*/

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlr4, 'select ''INFO: Table IAM_ROLES not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;
SET @LAST_ID_IN_IAM_ROLES = LAST_INSERT_ID();

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp41, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp42, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp43, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp44, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp45, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp46, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp47, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp48, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp49, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp410, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp411, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp412, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

 


/*
 business_user
*/

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlr5,'select ''INFO: Table IAM_ROLES not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;
SET @LAST_ID_IN_IAM_ROLES = LAST_INSERT_ID();

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp51, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp52, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp53, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp54, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp55, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp56, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp57, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp58, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp59, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp510, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp511, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;
 
 
/*
 api_user
*/

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlr6,'select ''INFO: Table IAM_ROLES not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;
SET @LAST_ID_IN_IAM_ROLES = LAST_INSERT_ID();

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp61, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp62, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp63, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp64, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp65, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp66, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp67, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp68, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp69, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp610, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @row_exists1 < 1 AND @row_exists2 < 1 AND @row_exists3 < 1 AND @row_exists4 < 1 AND @row_exists5 < 1,@sqlp611, 'select ''INFO: Table IAM_PERMISSIONS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;


COMMIT;


set SQL_MODE=ANSI_QUOTES;

/* set Security Level */
UPDATE INV_JS_INSTANCES SET "SECURITY_LEVEL" = 0 WHERE "SECURITY_LEVEL" <> 0;
COMMIT;
set SQL_MODE=ANSI_QUOTES;

/* INV_SEARCH_ITEMS */
/* Help table (see inventory_procedures.sql) - solution for MySQL 5.7 to extract JSON array values into multiple rows */
START TRANSACTION; 
set @count_rows := (SELECT count(*) FROM INV_SEARCH_ITEMS);

set @stmt := if( @count_rows < 1, 'INSERT INV_SEARCH_ITEMS VALUES (0),(1),(2),(3),(4),(5),(6),(7),(8),(9),(10),(11),(12),(13),(14),(15),(16),(17),(18),(19),(20),(21),(22),(23),(24),(25),(26),(27),(28),(29),(30),(31),(32),(33),(34),(35),(36),(37),(38),(39),(40),(41),(42),(43),(44),(45),(46),(47),(48),(49),(50),(51),(52),(53),(54),(55),(56),(57),(58),(59),(60),(61),(62),(63),(64),(65),(66),(67),(68),(69),(70),(71),(72),(73),(74),(75),(76),(77),(78),(79),(80),(81),(82),(83),(84),(85),(86),(87),(88),(89),(90),(91),(92),(93),(94),(95),(96),(97),(98),(99);', 'select ''INFO: INV_SEARCH_ITEMS rows found''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @count_rows < 1, 'INSERT INV_SEARCH_ITEMS VALUES (100),(101),(102),(103),(104),(105),(106),(107),(108),(109),(110),(111),(112),(113),(114),(115),(116),(117),(118),(119),(120),(121),(122),(123),(124),(125),(126),(127),(128),(129),(130),(131),(132),(133),(134),(135),(136),(137),(138),(139),(140),(141),(142),(143),(144),(145),(146),(147),(148),(149),(150),(151),(152),(153),(154),(155),(156),(157),(158),(159),(160),(161),(162),(163),(164),(165),(166),(167),(168),(169),(170),(171),(172),(173),(174),(175),(176),(177),(178),(179),(180),(181),(182),(183),(184),(185),(186),(187),(188),(189),(190),(191),(192),(193),(194),(195),(196),(197),(198),(199);', 'select ''INFO: INV_SEARCH_ITEMS rows found''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @count_rows < 1, 'INSERT INV_SEARCH_ITEMS VALUES (200),(201),(202),(203),(204),(205),(206),(207),(208),(209),(210),(211),(212),(213),(214),(215),(216),(217),(218),(219),(220),(221),(222),(223),(224),(225),(226),(227),(228),(229),(230),(231),(232),(233),(234),(235),(236),(237),(238),(239),(240),(241),(242),(243),(244),(245),(246),(247),(248),(249),(250),(251),(252),(253),(254),(255),(256),(257),(258),(259),(260),(261),(262),(263),(264),(265),(266),(267),(268),(269),(270),(271),(272),(273),(274),(275),(276),(277),(278),(279),(280),(281),(282),(283),(284),(285),(286),(287),(288),(289),(290),(291),(292),(293),(294),(295),(296),(297),(298),(299);', 'select ''INFO: INV_SEARCH_ITEMS rows found''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @count_rows < 1, 'INSERT INV_SEARCH_ITEMS VALUES (300),(301),(302),(303),(304),(305),(306),(307),(308),(309),(310),(311),(312),(313),(314),(315),(316),(317),(318),(319),(320),(321),(322),(323),(324),(325),(326),(327),(328),(329),(330),(331),(332),(333),(334),(335),(336),(337),(338),(339),(340),(341),(342),(343),(344),(345),(346),(347),(348),(349),(350),(351),(352),(353),(354),(355),(356),(357),(358),(359),(360),(361),(362),(363),(364),(365),(366),(367),(368),(369),(370),(371),(372),(373),(374),(375),(376),(377),(378),(379),(380),(381),(382),(383),(384),(385),(386),(387),(388),(389),(390),(391),(392),(393),(394),(395),(396),(397),(398),(399);', 'select ''INFO: INV_SEARCH_ITEMS rows found''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @count_rows < 1, 'INSERT INV_SEARCH_ITEMS VALUES (400),(401),(402),(403),(404),(405),(406),(407),(408),(409),(410),(411),(412),(413),(414),(415),(416),(417),(418),(419),(420),(421),(422),(423),(424),(425),(426),(427),(428),(429),(430),(431),(432),(433),(434),(435),(436),(437),(438),(439),(440),(441),(442),(443),(444),(445),(446),(447),(448),(449),(450),(451),(452),(453),(454),(455),(456),(457),(458),(459),(460),(461),(462),(463),(464),(465),(466),(467),(468),(469),(470),(471),(472),(473),(474),(475),(476),(477),(478),(479),(480),(481),(482),(483),(484),(485),(486),(487),(488),(489),(490),(491),(492),(493),(494),(495),(496),(497),(498),(499);', 'select ''INFO: INV_SEARCH_ITEMS rows found''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @count_rows < 1, 'INSERT INV_SEARCH_ITEMS VALUES (500),(501),(502),(503),(504),(505),(506),(507),(508),(509),(510),(511),(512),(513),(514),(515),(516),(517),(518),(519),(520),(521),(522),(523),(524),(525),(526),(527),(528),(529),(530),(531),(532),(533),(534),(535),(536),(537),(538),(539),(540),(541),(542),(543),(544),(545),(546),(547),(548),(549),(550),(551),(552),(553),(554),(555),(556),(557),(558),(559),(560),(561),(562),(563),(564),(565),(566),(567),(568),(569),(570),(571),(572),(573),(574),(575),(576),(577),(578),(579),(580),(581),(582),(583),(584),(585),(586),(587),(588),(589),(590),(591),(592),(593),(594),(595),(596),(597),(598),(599);', 'select ''INFO: INV_SEARCH_ITEMS rows found''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @count_rows < 1, 'INSERT INV_SEARCH_ITEMS VALUES (600),(601),(602),(603),(604),(605),(606),(607),(608),(609),(610),(611),(612),(613),(614),(615),(616),(617),(618),(619),(620),(621),(622),(623),(624),(625),(626),(627),(628),(629),(630),(631),(632),(633),(634),(635),(636),(637),(638),(639),(640),(641),(642),(643),(644),(645),(646),(647),(648),(649),(650),(651),(652),(653),(654),(655),(656),(657),(658),(659),(660),(661),(662),(663),(664),(665),(666),(667),(668),(669),(670),(671),(672),(673),(674),(675),(676),(677),(678),(679),(680),(681),(682),(683),(684),(685),(686),(687),(688),(689),(690),(691),(692),(693),(694),(695),(696),(697),(698),(699);', 'select ''INFO: INV_SEARCH_ITEMS rows found''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @count_rows < 1, 'INSERT INV_SEARCH_ITEMS VALUES (700),(701),(702),(703),(704),(705),(706),(707),(708),(709),(710),(711),(712),(713),(714),(715),(716),(717),(718),(719),(720),(721),(722),(723),(724),(725),(726),(727),(728),(729),(730),(731),(732),(733),(734),(735),(736),(737),(738),(739),(740),(741),(742),(743),(744),(745),(746),(747),(748),(749),(750),(751),(752),(753),(754),(755),(756),(757),(758),(759),(760),(761),(762),(763),(764),(765),(766),(767),(768),(769),(770),(771),(772),(773),(774),(775),(776),(777),(778),(779),(780),(781),(782),(783),(784),(785),(786),(787),(788),(789),(790),(791),(792),(793),(794),(795),(796),(797),(798),(799);', 'select ''INFO: INV_SEARCH_ITEMS rows found''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @count_rows < 1, 'INSERT INV_SEARCH_ITEMS VALUES (800),(801),(802),(803),(804),(805),(806),(807),(808),(809),(810),(811),(812),(813),(814),(815),(816),(817),(818),(819),(820),(821),(822),(823),(824),(825),(826),(827),(828),(829),(830),(831),(832),(833),(834),(835),(836),(837),(838),(839),(840),(841),(842),(843),(844),(845),(846),(847),(848),(849),(850),(851),(852),(853),(854),(855),(856),(857),(858),(859),(860),(861),(862),(863),(864),(865),(866),(867),(868),(869),(870),(871),(872),(873),(874),(875),(876),(877),(878),(879),(880),(881),(882),(883),(884),(885),(886),(887),(888),(889),(890),(891),(892),(893),(894),(895),(896),(897),(898),(899);', 'select ''INFO: INV_SEARCH_ITEMS rows found''');
PREPARE statement FROM @stmt;
EXECUTE statement;

set @stmt := if( @count_rows < 1, 'INSERT INV_SEARCH_ITEMS VALUES (900),(901),(902),(903),(904),(905),(906),(907),(908),(909),(910),(911),(912),(913),(914),(915),(916),(917),(918),(919),(920),(921),(922),(923),(924),(925),(926),(927),(928),(929),(930),(931),(932),(933),(934),(935),(936),(937),(938),(939),(940),(941),(942),(943),(944),(945),(946),(947),(948),(949),(950),(951),(952),(953),(954),(955),(956),(957),(958),(959),(960),(961),(962),(963),(964),(965),(966),(967),(968),(969),(970),(971),(972),(973),(974),(975),(976),(977),(978),(979),(980),(981),(982),(983),(984),(985),(986),(987),(988),(989),(990),(991),(992),(993),(994),(995),(996),(997),(998),(999);', 'select ''INFO: INV_SEARCH_ITEMS rows found''');
PREPARE statement FROM @stmt;
EXECUTE statement;

COMMIT;

/* DROP old table INV_JSON_ARRAY_HELPER */
DROP TABLE IF EXISTS INV_JSON_ARRAY_HELPER;

set SQL_MODE=ANSI_QUOTES;

/* delete/insert version */
DELETE FROM JOC_VARIABLES WHERE "NAME"='version';
INSERT INTO JOC_VARIABLES ("NAME", "NUMERIC_VALUE", "TEXT_VALUE", "BINARY_VALUE") VALUES ('version',NULL,'2.7.2-RC1',NULL);
COMMIT;
/*
 | Insert Script for default Keys for SecurityLevel LOW and MEDIUM
 | SOS GmbH, 2020-08-12
*/
set SQL_MODE=ANSI_QUOTES;

/*
 | ATTENTION:
 | The "SECLVL" field contains the security level (0=LOW, 1=MEDUIM, 2=HIGH).
 | This value MUST match the security level selected in the setup.
 | If the following two INSERT statements are to be executed before the setup, then the value of "SECLVL" may have to be adjusted.
 | 
 | set @row_exists := (SELECT EXISTS(SELECT * FROM DEP_KEYS ... AND "SECLVL"=[number of security level]));
 | set @sql := 'INSERT INTO DEP_KEYS( ... , [number of security level]);';
 | set @sql := 'INSERT INTO INV_CERTS( ... , [number of security level]);';
*/

/* import default */
set @row_exists := (SELECT EXISTS(SELECT * FROM DEP_KEYS WHERE ACCOUNT='root' AND SECLVL=0));
set @sqllm  := 'INSERT INTO DEP_KEYS("KEY_TYPE","KEY_ALG","KEY","CERTIFICATE","ACCOUNT","SECLVL") VALUES (0,2,''-----BEGIN EC PRIVATE KEY-----\nMHcCAQEEID+VjMgUjVph6n1lxu5Vx9qrhoCyMrr9vcmwKBx+MSG2oAoGCCqGSM49\nAwEHoUQDQgAEVN/5cTnhaX31K5Hh0N3sgMfKXY3JktHLvFxXTn2rPgtoKAef0NjB\nvSA5GJ7uM48zlyX3JraqSPDkeXJqLnQdww==\n-----END EC PRIVATE KEY-----'', ''-----BEGIN CERTIFICATE-----\nMIID1zCCAb+gAwIBAgIJAMxQSEdXteoRMA0GCSqGSIb3DQEBCwUAMIGNMQswCQYD\nVQQGEwJERTEPMA0GA1UECAwGQmVybGluMQ8wDQYDVQQHDAZCZXJsaW4xDDAKBgNV\nBAoMA1NPUzELMAkGA1UECwwCSVQxHDAaBgNVBAMME1NPUyBJbnRlcm1lZGlhdGUg\nQ0ExIzAhBgkqhkiG9w0BCQEWFGFkbWluQHNvcy1iZXJsaW4uY29tMB4XDTIwMTAx\nMzEyNTA1NVoXDTI1MTAxMjEyNTA1NVowbDELMAkGA1UEBhMCREUxDzANBgNVBAgM\nBkJlcmxpbjEPMA0GA1UEBwwGQmVybGluMQwwCgYDVQQKDANTT1MxCzAJBgNVBAsM\nAklUMQ8wDQYDVQQDDAZzb3MtZWMxDzANBgNVBC4TBlNPUyBDQTBZMBMGByqGSM49\nAgEGCCqGSM49AwEHA0IABFTf+XE54Wl99SuR4dDd7IDHyl2NyZLRy7xcV059qz4L\naCgHn9DYwb0gORie7jOPM5cl9ya2qkjw5Hlyai50HcOjJTAjMBEGCWCGSAGG+EIB\nAQQEAwIHgDAOBgNVHQ8BAf8EBAMCBeAwDQYJKoZIhvcNAQELBQADggIBAGC7BKCd\nkuhIzEGflG4tslnWlC7BXvbI2lKkUbDYlcBf04hYl4ootWfprFfF80LXBFgWUxGN\n6HZIQDwWJch/h+UFjf/97zVxzbl0q1miZtNN2USbYK3yfrgWyOKEzZcAxUnr2pL4\n9/t2LBmFKvgmh1RiB40Vf8s5frEQID2Am+I+pZsi4R7/OtwUo6dZJKP6IFlqZYzf\nklWlLMgwVK6Iqo+Lu8M2oxWRHxMCTOAQyQmVWf8ZsfzEn3k412v5tejWjQsoo057\nVCQ3p36UOl2+Qrif2PvYOz3pvdek6xwuJaqpr1fDSjEO/WSpl4Fdx0pavCFLniaN\nodP00FpH1OfwHf3XvYMCcHO+/JRlsc4uNaalEkD6n3pNbWlpJyaneEUr7GqC8TEX\nzHJ3r5SSKFxwEX/iAL5CyIgq8BUFLQtXFBiYVTMQXyUTNixo0h130bV4KMdpfLLm\nu+iekRBGM59sP5Ijfg4lvQqoMZ2Ck+BGBye/tlItgDOIh+FIIVMwfHY4KOlfaqRr\nJRzEpVkssoIVHNghl1kT3U9ZSbMtJ3W650uBtidhS0tYOXPixd5DJ8VI5cl16pwb\nDpkAS01bxgDhQrvQINzrLTXeDuY01Lr3HC8TD/NMS+p7X5CQ1KxUhgzxaPQ8LTau\n9DzuuCRcbkIBI+KQl5KF8kZ2ohdw/tNOH+RK\n-----END CERTIFICATE-----'', ''root'', 0);';
set @sqlh   := 'INSERT INTO DEP_KEYS("KEY_TYPE","KEY_ALG","KEY","CERTIFICATE","ACCOUNT","SECLVL") VALUES (1,2,null, ''-----BEGIN CERTIFICATE-----\nMIID1zCCAb+gAwIBAgIJAMxQSEdXteoRMA0GCSqGSIb3DQEBCwUAMIGNMQswCQYD\nVQQGEwJERTEPMA0GA1UECAwGQmVybGluMQ8wDQYDVQQHDAZCZXJsaW4xDDAKBgNV\nBAoMA1NPUzELMAkGA1UECwwCSVQxHDAaBgNVBAMME1NPUyBJbnRlcm1lZGlhdGUg\nQ0ExIzAhBgkqhkiG9w0BCQEWFGFkbWluQHNvcy1iZXJsaW4uY29tMB4XDTIwMTAx\nMzEyNTA1NVoXDTI1MTAxMjEyNTA1NVowbDELMAkGA1UEBhMCREUxDzANBgNVBAgM\nBkJlcmxpbjEPMA0GA1UEBwwGQmVybGluMQwwCgYDVQQKDANTT1MxCzAJBgNVBAsM\nAklUMQ8wDQYDVQQDDAZzb3MtZWMxDzANBgNVBC4TBlNPUyBDQTBZMBMGByqGSM49\nAgEGCCqGSM49AwEHA0IABFTf+XE54Wl99SuR4dDd7IDHyl2NyZLRy7xcV059qz4L\naCgHn9DYwb0gORie7jOPM5cl9ya2qkjw5Hlyai50HcOjJTAjMBEGCWCGSAGG+EIB\nAQQEAwIHgDAOBgNVHQ8BAf8EBAMCBeAwDQYJKoZIhvcNAQELBQADggIBAGC7BKCd\nkuhIzEGflG4tslnWlC7BXvbI2lKkUbDYlcBf04hYl4ootWfprFfF80LXBFgWUxGN\n6HZIQDwWJch/h+UFjf/97zVxzbl0q1miZtNN2USbYK3yfrgWyOKEzZcAxUnr2pL4\n9/t2LBmFKvgmh1RiB40Vf8s5frEQID2Am+I+pZsi4R7/OtwUo6dZJKP6IFlqZYzf\nklWlLMgwVK6Iqo+Lu8M2oxWRHxMCTOAQyQmVWf8ZsfzEn3k412v5tejWjQsoo057\nVCQ3p36UOl2+Qrif2PvYOz3pvdek6xwuJaqpr1fDSjEO/WSpl4Fdx0pavCFLniaN\nodP00FpH1OfwHf3XvYMCcHO+/JRlsc4uNaalEkD6n3pNbWlpJyaneEUr7GqC8TEX\nzHJ3r5SSKFxwEX/iAL5CyIgq8BUFLQtXFBiYVTMQXyUTNixo0h130bV4KMdpfLLm\nu+iekRBGM59sP5Ijfg4lvQqoMZ2Ck+BGBye/tlItgDOIh+FIIVMwfHY4KOlfaqRr\nJRzEpVkssoIVHNghl1kT3U9ZSbMtJ3W650uBtidhS0tYOXPixd5DJ8VI5cl16pwb\nDpkAS01bxgDhQrvQINzrLTXeDuY01Lr3HC8TD/NMS+p7X5CQ1KxUhgzxaPQ8LTau\n9DzuuCRcbkIBI+KQl5KF8kZ2ohdw/tNOH+RK\n-----END CERTIFICATE-----'', ''root'', 0);';
set @stmtlm := if( @row_exists < 1,@sqllm, 'select ''INFO: Table DEP_KEYS not found!''');
set @stmth  := if( @row_exists < 1,@sqlh, 'select ''INFO: Table DEP_KEYS not found!''');
set @stmt   := if(0 < 2, @stmtlm, @stmth);
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;

set @row_exists := (SELECT EXISTS(SELECT * FROM INV_CERTS WHERE ACCOUNT='root' AND CA=1));
set @sql := 'INSERT INTO INV_CERTS("KEY_TYPE","KEY_ALG","PEM", "CA","ACCOUNT","SECLVL") VALUES (1,1,''-----BEGIN CERTIFICATE-----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\nHwphnWeOC52Ovuw2OLv1TomJKSzujvLzpA==\n-----END CERTIFICATE-----'', 1, ''root'', 0);';
set @stmt := if( @row_exists < 1,@sql, 'select ''INFO: Table DEP_KEYS not found!''');
PREPARE statement FROM @stmt;
EXECUTE statement;
COMMIT;
/*
 | Deployment views for MySQL
 | SOS GmbH, 2019-10-04
*/
set SQL_MODE=ANSI_QUOTES;

/* View for current inventory at the controllers */
CREATE OR REPLACE VIEW DEP_CONFIGURATIONS_JOIN AS SELECT
    "CONTROLLER_ID",
    "NAME",
    "TYPE",
    MAX("ID") AS "MAX_ID"
FROM DEP_HISTORY
WHERE "STATE" = 0
GROUP BY
    "CONTROLLER_ID",
    "NAME",
    "TYPE";

CREATE OR REPLACE VIEW DEP_CONFIGURATIONS AS SELECT
    a."ID",
    a."NAME",
    a."TITLE",
    a."PATH",
    a."FOLDER",
    a."TYPE",
    a."INV_CID",
    a."CONTROLLER_ID",
    a."INV_CONTENT" AS "CONTENT",
    a."COMMIT_ID",
    a."DEPLOYMENT_DATE" AS "CREATED"
FROM DEP_HISTORY AS a
INNER JOIN DEP_CONFIGURATIONS_JOIN AS b
    ON a."ID" = b."MAX_ID"
    WHERE a."OPERATION" = 0;

CREATE OR REPLACE VIEW DEP_NAMEPATHS AS SELECT
    a."ID",
    a."NAME",
    a."PATH",
    a."TYPE",
    a."CONTROLLER_ID"
FROM DEP_HISTORY AS a
INNER JOIN DEP_CONFIGURATIONS_JOIN AS b
    ON a."ID" = b."MAX_ID";
/*
 | Inventory views for MySQL
*/
set SQL_MODE=ANSI_QUOTES;

CREATE OR REPLACE VIEW INV_SCHEDULE2CALENDARS AS 
    SELECT 
        JSON_UNQUOTE(JSON_EXTRACT(i."JSON_CONTENT",CONCAT('$.calendars[', h."ROW", '].calendarName'))) AS "CALENDAR_NAME",
        i."NAME"            AS "SCHEDULE_NAME",
        i."PATH"            AS "SCHEDULE_PATH",
        i."FOLDER"          AS "SCHEDULE_FOLDER",
        i."RELEASED"        AS "SCHEDULE_RELEASED"
    FROM INV_CONFIGURATIONS i
        INNER JOIN INV_SEARCH_ITEMS h ON h."ROW" < JSON_LENGTH(i."JSON_CONTENT",'$.calendars')
    WHERE i."TYPE"=7
        AND JSON_UNQUOTE(JSON_EXTRACT(i."JSON_CONTENT",'$.calendars')) IS NOT NULL;
        
CREATE OR REPLACE VIEW INV_REL_SCHEDULE2CALENDARS AS 
    SELECT 
        JSON_UNQUOTE(JSON_EXTRACT(i."JSON_CONTENT",CONCAT('$.calendars[', h."ROW", '].calendarName'))) AS "CALENDAR_NAME",
        i."NAME"            AS "SCHEDULE_NAME",
        i."PATH"            AS "SCHEDULE_PATH",
        i."FOLDER"          AS "SCHEDULE_FOLDER"
    FROM INV_RELEASED_CONFIGURATIONS i
        INNER JOIN INV_SEARCH_ITEMS h ON h."ROW" < JSON_LENGTH(i."JSON_CONTENT",'$.calendars')
    WHERE i."TYPE"=7
        AND JSON_UNQUOTE(JSON_EXTRACT(i."JSON_CONTENT",'$.calendars')) IS NOT NULL;
/*
 | Inventory views for MySQL
*/
set SQL_MODE=ANSI_QUOTES;

CREATE OR REPLACE VIEW INV_SCHEDULE2WORKFLOWS AS 
    SELECT
        JSON_UNQUOTE(JSON_EXTRACT("JSON_CONTENT",'$.workflowName')) AS "WORKFLOW_NAME",
        "NAME"          AS "SCHEDULE_NAME",
        "PATH"          AS "SCHEDULE_PATH",
        "FOLDER"        AS "SCHEDULE_FOLDER",
        "RELEASED"      AS "SCHEDULE_RELEASED",
        "JSON_CONTENT"  AS "SCHEDULE_CONTENT"
    FROM INV_CONFIGURATIONS
    WHERE "TYPE"=7
        AND JSON_UNQUOTE(JSON_EXTRACT("JSON_CONTENT",'$.workflowNames')) IS NULL
    UNION ALL
    SELECT 
        JSON_UNQUOTE(JSON_EXTRACT(i."JSON_CONTENT",CONCAT('$.workflowNames[', h."ROW", ']'))) AS "WORKFLOW_NAME",
        i."NAME"            AS "SCHEDULE_NAME",
        i."PATH"            AS "SCHEDULE_PATH",
        i."FOLDER"          AS "SCHEDULE_FOLDER",
        i."RELEASED"        AS "SCHEDULE_RELEASED",
        i."JSON_CONTENT"    AS "SCHEDULE_CONTENT"
    FROM INV_CONFIGURATIONS i
        INNER JOIN INV_SEARCH_ITEMS h ON h."ROW" < JSON_LENGTH(i."JSON_CONTENT",'$.workflowNames')
    WHERE i."TYPE"=7
        AND JSON_UNQUOTE(JSON_EXTRACT(i."JSON_CONTENT",'$.workflowNames')) IS NOT NULL;
        
DROP VIEW IF EXISTS INV_RELEASED_SCHEDULE2WORKFLOWS;        
CREATE OR REPLACE VIEW INV_REL_SCHEDULE2WORKFLOWS AS 
    SELECT
        JSON_UNQUOTE(JSON_EXTRACT("JSON_CONTENT",'$.workflowName')) AS "WORKFLOW_NAME",
        "NAME"          AS "SCHEDULE_NAME",
        "PATH"          AS "SCHEDULE_PATH",
        "FOLDER"        AS "SCHEDULE_FOLDER",
        "JSON_CONTENT"  AS "SCHEDULE_CONTENT"
    FROM INV_RELEASED_CONFIGURATIONS
    WHERE "TYPE"=7
        AND JSON_UNQUOTE(JSON_EXTRACT("JSON_CONTENT",'$.workflowNames')) IS NULL
    UNION ALL
    SELECT 
        JSON_UNQUOTE(JSON_EXTRACT(i."JSON_CONTENT",CONCAT('$.workflowNames[', h."ROW", ']'))) AS "WORKFLOW_NAME",
        i."NAME"            AS "SCHEDULE_NAME",
        i."PATH"            AS "SCHEDULE_PATH",
        i."FOLDER"          AS "SCHEDULE_FOLDER",
        i."JSON_CONTENT"    AS "SCHEDULE_CONTENT"
    FROM INV_RELEASED_CONFIGURATIONS i
        INNER JOIN INV_SEARCH_ITEMS h ON h."ROW" < JSON_LENGTH(i."JSON_CONTENT",'$.workflowNames')
    WHERE i."TYPE"=7
        AND JSON_UNQUOTE(JSON_EXTRACT(i."JSON_CONTENT",'$.workflowNames')) IS NOT NULL;
