/*
 * Decompiled with CFR 0.152.
 */
package org.linguafranca.pwdb.kdbx;

import com.google.common.io.LittleEndianDataInputStream;
import com.google.common.io.LittleEndianDataOutputStream;
import java.io.DataInput;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.charset.StandardCharsets;
import java.security.DigestInputStream;
import java.security.DigestOutputStream;
import java.security.MessageDigest;
import java.util.Arrays;
import java.util.Map;
import java.util.function.IntConsumer;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;
import javax.crypto.Mac;
import org.linguafranca.pwdb.Credentials;
import org.linguafranca.pwdb.hashedblock.CollectingInputStream;
import org.linguafranca.pwdb.hashedblock.CollectingOutputStream;
import org.linguafranca.pwdb.hashedblock.HashedBlockInputStream;
import org.linguafranca.pwdb.hashedblock.HashedBlockOutputStream;
import org.linguafranca.pwdb.hashedblock.HmacBlockInputStream;
import org.linguafranca.pwdb.hashedblock.HmacBlockOutputStream;
import org.linguafranca.pwdb.kdbx.Helpers;
import org.linguafranca.pwdb.kdbx.KdbxHeader;
import org.linguafranca.pwdb.security.Encryption;
import org.linguafranca.pwdb.security.VariantDictionary;

public class KdbxSerializer {
    private static final int SIG1 = -1700603645;
    private static final int SIG2 = -1253311641;
    private static final int FILE_VERSION_32 = 196609;
    private static final int FILE_VERSION_4 = 262144;

    private KdbxSerializer() {
    }

    public static InputStream createUnencryptedInputStream(Credentials credentials, KdbxHeader kdbxHeader, InputStream inputStream) throws IOException {
        InputStream plainTextStream;
        KdbxSerializer.readOuterHeader(inputStream, kdbxHeader);
        if (kdbxHeader.getVersion() >= 4) {
            KdbxSerializer.readOuterHeaderVerification(kdbxHeader, credentials, new DataInputStream(inputStream));
            HmacBlockInputStream hmacBlockInputStream = new HmacBlockInputStream(kdbxHeader.getHmacKey(credentials), inputStream, true);
            plainTextStream = kdbxHeader.createDecryptedStream(credentials.getKey(), hmacBlockInputStream);
        } else {
            InputStream decryptedInputStream = kdbxHeader.createDecryptedStream(credentials.getKey(), inputStream);
            KdbxSerializer.checkStartBytes(kdbxHeader, decryptedInputStream);
            plainTextStream = new HashedBlockInputStream(decryptedInputStream, true);
        }
        if (kdbxHeader.getCompressionFlags().equals((Object)KdbxHeader.CompressionFlags.GZIP)) {
            plainTextStream = new GZIPInputStream(plainTextStream);
        }
        if (kdbxHeader.getVersion() >= 4) {
            KdbxSerializer.readInnerHeader(kdbxHeader, plainTextStream);
        }
        return plainTextStream;
    }

    public static OutputStream createEncryptedOutputStream(Credentials credentials, KdbxHeader kdbxHeader, OutputStream outputStream) throws IOException {
        OutputStream result;
        KdbxSerializer.writeKdbxHeader(kdbxHeader, outputStream);
        if (kdbxHeader.getVersion() >= 4) {
            KdbxSerializer.writeOuterHeaderVerification(kdbxHeader, credentials, new DataOutputStream(outputStream));
            HmacBlockOutputStream blockOutputStream = new HmacBlockOutputStream(kdbxHeader.getHmacKey(credentials), outputStream, true);
            result = kdbxHeader.createEncryptedStream(credentials.getKey(), blockOutputStream);
        } else {
            OutputStream encryptedOutputStream = kdbxHeader.createEncryptedStream(credentials.getKey(), outputStream);
            KdbxSerializer.writeStartBytes(kdbxHeader, encryptedOutputStream);
            result = new HashedBlockOutputStream(encryptedOutputStream, true);
        }
        if (kdbxHeader.getCompressionFlags().equals((Object)KdbxHeader.CompressionFlags.GZIP)) {
            result = new GZIPOutputStream(result);
        }
        if (kdbxHeader.getVersion() >= 4) {
            KdbxSerializer.writeInnerHeader(kdbxHeader, result);
        }
        return result;
    }

    private static void checkStartBytes(KdbxHeader kdbxHeader, InputStream decryptedInputStream) throws IOException {
        LittleEndianDataInputStream ledis = new LittleEndianDataInputStream(decryptedInputStream);
        byte[] startBytes = new byte[32];
        ledis.readFully(startBytes);
        if (!Arrays.equals(startBytes, kdbxHeader.getStreamStartBytes())) {
            throw new IllegalStateException("Inconsistent stream start bytes. This usually means the credentials were wrong.");
        }
    }

    private static void writeStartBytes(KdbxHeader kdbxHeader, OutputStream encryptedOutputStream) throws IOException {
        LittleEndianDataOutputStream ledos = new LittleEndianDataOutputStream(encryptedOutputStream);
        ledos.write(kdbxHeader.getStreamStartBytes());
    }

    public static KdbxHeader readOuterHeader(InputStream inputStream, KdbxHeader kdbxHeader) throws IOException {
        MessageDigest digest = Encryption.getSha256MessageDigestInstance();
        DigestInputStream shaDigestInputStream = new DigestInputStream(inputStream, digest);
        CollectingInputStream collectingInputStream = new CollectingInputStream(shaDigestInputStream, true);
        LittleEndianDataInputStream ledis = new LittleEndianDataInputStream((InputStream)collectingInputStream);
        if (!KdbxSerializer.verifyMagicNumber(ledis)) {
            throw new IllegalStateException("Magic number did not match");
        }
        int fullVersion = ledis.readInt();
        kdbxHeader.setVersion(fullVersion >> 16);
        KdbxSerializer.getOuterHeaderFields(kdbxHeader, (DataInput)ledis);
        shaDigestInputStream.on(false);
        collectingInputStream.setCollecting(false);
        kdbxHeader.setHeaderHash(digest.digest());
        kdbxHeader.setHeaderBytes(collectingInputStream.getCollectedBytes());
        return kdbxHeader;
    }

    public static void readOuterHeaderVerification(KdbxHeader kdbxHeader, Credentials credentials, DataInput input) throws IOException {
        byte[] receivedHmacSha256;
        byte[] receivedSha256 = KdbxSerializer.getBytes(32, input);
        if (!Arrays.equals(kdbxHeader.getHeaderHash(), receivedSha256)) {
            throw new IllegalStateException("Header hash does not match");
        }
        byte[] hmacKey = kdbxHeader.getHmacKey(credentials);
        byte[] hmacKey64 = Encryption.transformHmacKey((byte[])hmacKey, (byte[])Helpers.toBytes(-1L, ByteOrder.LITTLE_ENDIAN));
        Mac mac = Encryption.getHMacSha256Instance((byte[])hmacKey64);
        byte[] computedHmacSha256 = mac.doFinal(kdbxHeader.getHeaderBytes());
        if (!Arrays.equals(computedHmacSha256, receivedHmacSha256 = KdbxSerializer.getBytes(32, input))) {
            throw new IllegalStateException("Header HMAC does not match");
        }
    }

    private static void writeOuterHeaderVerification(KdbxHeader kdbxHeader, Credentials credentials, DataOutputStream dataOutputStream) throws IOException {
        dataOutputStream.write(kdbxHeader.getHeaderHash());
        byte[] hmacKey = kdbxHeader.getHmacKey(credentials);
        byte[] hmacKey64 = Encryption.transformHmacKey((byte[])hmacKey, (byte[])Helpers.toBytes(-1L, ByteOrder.LITTLE_ENDIAN));
        Mac mac = Encryption.getHMacSha256Instance((byte[])hmacKey64);
        byte[] hashedHeaderBytes = mac.doFinal(kdbxHeader.getHeaderBytes());
        dataOutputStream.write(hashedHeaderBytes);
    }

    private static void getOuterHeaderFields(KdbxHeader kdbxHeader, DataInput input) throws IOException {
        byte headerType;
        do {
            headerType = input.readByte();
            int length = kdbxHeader.getVersion() == 3 ? input.readShort() : input.readInt();
            switch (headerType) {
                case 0: {
                    KdbxSerializer.getBytes(length, input);
                    break;
                }
                case 1: {
                    KdbxSerializer.getBytes(length, input);
                    break;
                }
                case 2: {
                    kdbxHeader.setCipherUuid(KdbxSerializer.getBytes(length, input));
                    break;
                }
                case 3: {
                    kdbxHeader.setCompressionFlags(KdbxSerializer.getInt(length, input));
                    break;
                }
                case 4: {
                    kdbxHeader.setMasterSeed(KdbxSerializer.getBytes(length, input));
                    break;
                }
                case 5: {
                    kdbxHeader.setTransformSeed(KdbxSerializer.getBytes(length, input));
                    break;
                }
                case 6: {
                    kdbxHeader.setTransformRounds(KdbxSerializer.getLong(length, input));
                    break;
                }
                case 7: {
                    kdbxHeader.setEncryptionIv(KdbxSerializer.getBytes(length, input));
                    break;
                }
                case 8: {
                    kdbxHeader.setInnerRandomStreamKey(KdbxSerializer.getBytes(length, input));
                    break;
                }
                case 9: {
                    kdbxHeader.setStreamStartBytes(KdbxSerializer.getBytes(length, input));
                    break;
                }
                case 10: {
                    kdbxHeader.setInnerRandomStreamId(KdbxSerializer.getInt(length, input));
                    break;
                }
                case 11: {
                    kdbxHeader.setKdfParameters(KdbxSerializer.readVariantDictionary(KdbxSerializer.getBytes(length, input)));
                    break;
                }
                case 12: {
                    kdbxHeader.setCustomData(KdbxSerializer.readVariantDictionary(KdbxSerializer.getBytes(length, input)));
                    break;
                }
                default: {
                    throw new IllegalStateException("Unknown File Header");
                }
            }
        } while (headerType != 0);
    }

    public static void writeKdbxHeader(KdbxHeader kdbxHeader, OutputStream outputStream) throws IOException {
        MessageDigest messageDigest = Encryption.getSha256MessageDigestInstance();
        DigestOutputStream digestOutputStream = new DigestOutputStream(outputStream, messageDigest);
        CollectingOutputStream collectingOutputStream = new CollectingOutputStream(digestOutputStream);
        LittleEndianDataOutputStream ledos = new LittleEndianDataOutputStream((OutputStream)collectingOutputStream);
        IntConsumer lengthWriter = i -> {
            try {
                if (kdbxHeader.getVersion() == 3) {
                    ledos.writeShort(i);
                } else {
                    ledos.writeInt(i);
                }
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
        };
        ledos.writeInt(-1700603645);
        ledos.writeInt(-1253311641);
        ledos.writeInt(kdbxHeader.getVersion() == 3 ? 196609 : 262144);
        ledos.writeByte(2);
        lengthWriter.accept(16);
        byte[] b = new byte[16];
        ByteBuffer bb = ByteBuffer.wrap(b);
        bb.putLong(kdbxHeader.getCipherUuid().getMostSignificantBits());
        bb.putLong(8, kdbxHeader.getCipherUuid().getLeastSignificantBits());
        ledos.write(b);
        ledos.writeByte(3);
        lengthWriter.accept(4);
        ledos.writeInt(kdbxHeader.getCompressionFlags().ordinal());
        ledos.writeByte(4);
        lengthWriter.accept(kdbxHeader.getMasterSeed().length);
        ledos.write(kdbxHeader.getMasterSeed());
        if (kdbxHeader.getVersion() < 4) {
            ledos.writeByte(5);
            lengthWriter.accept(kdbxHeader.getTransformSeed().length);
            ledos.write(kdbxHeader.getTransformSeed());
            ledos.writeByte(6);
            lengthWriter.accept(8);
            ledos.writeLong(kdbxHeader.getTransformRounds());
        }
        ledos.writeByte(7);
        lengthWriter.accept(kdbxHeader.getEncryptionIv().length);
        ledos.write(kdbxHeader.getEncryptionIv());
        if (kdbxHeader.getVersion() < 4) {
            ledos.writeByte(8);
            lengthWriter.accept(kdbxHeader.getInnerRandomStreamKey().length);
            ledos.write(kdbxHeader.getInnerRandomStreamKey());
            ledos.writeByte(9);
            lengthWriter.accept(kdbxHeader.getStreamStartBytes().length);
            ledos.write(kdbxHeader.getStreamStartBytes());
            ledos.writeByte(10);
            lengthWriter.accept(4);
            ledos.writeInt(kdbxHeader.getProtectedStreamAlgorithm().ordinal());
        }
        if (kdbxHeader.getVersion() > 3) {
            ledos.writeByte(11);
            byte[] vd = KdbxSerializer.serializeVariantDictionary(kdbxHeader.getKdfParameters());
            lengthWriter.accept(vd.length);
            ledos.write(vd);
        }
        ledos.writeByte(0);
        lengthWriter.accept(0);
        MessageDigest digest = digestOutputStream.getMessageDigest();
        kdbxHeader.setHeaderHash(digest.digest());
        collectingOutputStream.setCollecting(false);
        kdbxHeader.setHeaderBytes(collectingOutputStream.getCollectedBytes());
    }

    private static void readInnerHeader(KdbxHeader kdbxHeader, InputStream plainTextStream) throws IOException {
        byte headerType;
        LittleEndianDataInputStream input = new LittleEndianDataInputStream(plainTextStream);
        do {
            headerType = input.readByte();
            int length = input.readInt();
            switch (headerType) {
                case 0: {
                    KdbxSerializer.getBytes(length, (DataInput)input);
                    break;
                }
                case 1: {
                    kdbxHeader.setInnerRandomStreamId(KdbxSerializer.getInt(length, (DataInput)input));
                    break;
                }
                case 2: {
                    kdbxHeader.setInnerRandomStreamKey(KdbxSerializer.getBytes(length, (DataInput)input));
                    break;
                }
                case 3: {
                    kdbxHeader.addBinary(KdbxSerializer.getBytes(length, (DataInput)input));
                    break;
                }
                default: {
                    throw new IllegalStateException("Invalid inner header field");
                }
            }
        } while (headerType != 0);
    }

    public static void writeInnerHeader(KdbxHeader kdbxHeader, OutputStream outputStream) throws IOException {
        LittleEndianDataOutputStream output = new LittleEndianDataOutputStream(outputStream);
        output.writeByte(1);
        output.writeInt(4);
        output.writeInt(kdbxHeader.getProtectedStreamAlgorithm().ordinal());
        output.writeByte(2);
        output.writeInt(kdbxHeader.getInnerRandomStreamKey().length);
        output.write(kdbxHeader.getInnerRandomStreamKey());
        for (byte[] binary : kdbxHeader.getBinaries()) {
            output.writeByte(3);
            output.writeInt(binary.length);
            output.write(binary);
        }
        output.writeByte(0);
        output.writeInt(0);
    }

    public static VariantDictionary readVariantDictionary(byte[] source) {
        ByteBuffer buf = ByteBuffer.wrap(source);
        buf.order(ByteOrder.LITTLE_ENDIAN);
        VariantDictionary vd = new VariantDictionary((short)(buf.getShort() >> 8));
        byte type = buf.get();
        while (type != 0) {
            int keyLength = buf.getInt();
            byte[] key = new byte[keyLength];
            buf.get(key);
            int valueLength = buf.getInt();
            byte[] value = new byte[valueLength];
            buf.get(value);
            vd.put(new String(key, StandardCharsets.US_ASCII), VariantDictionary.EntryType.get((byte)type), value);
            type = buf.get();
        }
        return vd;
    }

    public static byte[] serializeVariantDictionary(VariantDictionary v) {
        ByteBuffer buf = ByteBuffer.wrap(new byte[1024]);
        buf.order(ByteOrder.LITTLE_ENDIAN);
        buf.mark();
        buf.putShort((short)256);
        for (Map.Entry e : v.getEntries().entrySet()) {
            buf.put(((VariantDictionary.Entry)e.getValue()).getType());
            buf.putInt(((String)e.getKey()).length());
            buf.put(((String)e.getKey()).getBytes(StandardCharsets.US_ASCII));
            buf.putInt(((VariantDictionary.Entry)e.getValue()).asByteArray().length);
            buf.put(((VariantDictionary.Entry)e.getValue()).asByteArray());
        }
        buf.put((byte)0);
        byte[] result = new byte[buf.position()];
        buf.reset();
        buf.get(result);
        return result;
    }

    private static boolean verifyMagicNumber(LittleEndianDataInputStream ledis) throws IOException {
        int sig1 = ledis.readInt();
        int sig2 = ledis.readInt();
        return sig1 == -1700603645 && sig2 == -1253311641;
    }

    private static int getInt(int length, DataInput input) throws IOException {
        if (length != 4) {
            throw new IllegalStateException("Int required but length was " + length);
        }
        return input.readInt();
    }

    private static long getLong(int length, DataInput input) throws IOException {
        if (length != 8) {
            throw new IllegalStateException("Long required but length was " + length);
        }
        return input.readLong();
    }

    private static byte[] getBytes(int numBytes, DataInput input) throws IOException {
        byte[] result = new byte[numBytes];
        input.readFully(result);
        return result;
    }

    private static class InnerHeaderType {
        private static final byte END = 0;
        private static final byte INNER_RANDOM_STREAM_ID = 1;
        private static final byte INNER_RANDOM_STREAM_KEY = 2;
        private static final byte BINARY = 3;

        private InnerHeaderType() {
        }
    }

    private static class HeaderType {
        static final byte END = 0;
        static final byte COMMENT = 1;
        static final byte CIPHER_ID = 2;
        static final byte COMPRESSION_FLAGS = 3;
        static final byte MASTER_SEED = 4;
        static final byte TRANSFORM_SEED = 5;
        static final byte TRANSFORM_ROUNDS = 6;
        static final byte ENCRYPTION_IV = 7;
        static final byte INNER_RANDOM_STREAM_KEY = 8;
        static final byte STREAM_START_BYTES = 9;
        static final byte INNER_RANDOM_STREAM_ID = 10;
        static final byte KDF_PARAMETERS = 11;
        static final byte CUSTOM_DATA = 12;

        private HeaderType() {
        }
    }
}

