/*
 * Decompiled with CFR 0.152.
 */
package com.sos.commons.sign.keys.certificate;

import java.io.IOException;
import java.security.cert.CertificateEncodingException;
import java.security.cert.CertificateExpiredException;
import java.security.cert.CertificateNotYetValidException;
import java.security.cert.CertificateParsingException;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Base64;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;
import javax.naming.InvalidNameException;
import javax.naming.ldap.LdapName;
import org.bouncycastle.asn1.ASN1Encodable;
import org.bouncycastle.asn1.x500.AttributeTypeAndValue;
import org.bouncycastle.asn1.x500.RDN;
import org.bouncycastle.asn1.x500.X500Name;
import org.bouncycastle.asn1.x500.style.BCStyle;
import org.bouncycastle.asn1.x500.style.IETFUtils;
import org.bouncycastle.cert.jcajce.JcaX509CertificateHolder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class CertificateUtils {
    private static final Logger LOGGER = LoggerFactory.getLogger(CertificateUtils.class);

    public static String getCommonName(X509Certificate cert) throws InvalidNameException {
        LdapName ldapName = new LdapName(cert.getSubjectX500Principal().getName());
        return ldapName.getRdns().stream().filter(rdn -> rdn.getType().equalsIgnoreCase("CN")).findFirst().get().getValue().toString();
    }

    public static void logCertificateInfo(X509Certificate certificate) {
        if (certificate != null) {
            Set<String> nonCriticalExtensions;
            try {
                certificate.checkValidity();
                LOGGER.info("X509 certificate is valid.");
            }
            catch (CertificateExpiredException e) {
                LOGGER.info("X509 certificate has expired.");
            }
            catch (CertificateNotYetValidException e) {
                LOGGER.info("X509 certificate not yet valid.");
            }
            LOGGER.info("x509 certificate valid from - " + certificate.getNotBefore() + " - until - " + certificate.getNotAfter());
            String licenceCheckMessage = "";
            if (!licenceCheckMessage.isEmpty()) {
                LOGGER.info(licenceCheckMessage);
            }
            LOGGER.info("Subject DN: " + certificate.getSubjectX500Principal().getName());
            try {
                LOGGER.info("CN: " + CertificateUtils.getCommonName(certificate));
            }
            catch (InvalidNameException e) {
                LOGGER.error(e.getMessage(), (Throwable)e);
            }
            LOGGER.info("Issuer DN: " + certificate.getIssuerX500Principal().getName());
            LOGGER.info("Version: v" + certificate.getVersion());
            LOGGER.info("Type: " + certificate.getType());
            LOGGER.info("Serial Number: " + certificate.getSerialNumber());
            LOGGER.info("signature Algorithm Name: " + certificate.getSigAlgName());
            LOGGER.info("signature Algorithm OID: " + certificate.getSigAlgOID());
            Set<String> criticalExtensions = certificate.getCriticalExtensionOIDs();
            if (!criticalExtensions.isEmpty()) {
                LOGGER.info("Active Critical Extensions:");
                criticalExtensions.stream().forEach(item -> {
                    if ("2.5.29.15".equals(item)) {
                        LOGGER.info("  Key Usage (OID: 2.5.29.15)");
                        boolean[] keyUsages = certificate.getKeyUsage();
                        if (keyUsages[0]) {
                            LOGGER.info("    digitalSignature        (0)");
                        }
                        if (keyUsages[1]) {
                            LOGGER.info("    nonRepudiation          (1)");
                        }
                        if (keyUsages[2]) {
                            LOGGER.info("    keyEncipherment         (2)");
                        }
                        if (keyUsages[3]) {
                            LOGGER.info("    dataEncipherment        (3)");
                        }
                        if (keyUsages[4]) {
                            LOGGER.info("    keyAgreement            (4)");
                        }
                        if (keyUsages[5]) {
                            LOGGER.info("    keyCertSign             (5)");
                        }
                        if (keyUsages[6]) {
                            LOGGER.info("    cRLSign                 (6)");
                        }
                        if (keyUsages[7]) {
                            LOGGER.info("    encipherOnly            (7)");
                        }
                        if (keyUsages[8]) {
                            LOGGER.info("    decipherOnly            (8)");
                        }
                    } else if ("2.5.29.19".equals(item)) {
                        LOGGER.info("  Basic Constraints (OID: 2.5.29.19)");
                        int pathLenConstraint = certificate.getBasicConstraints();
                        if (pathLenConstraint != -1) {
                            LOGGER.info("    is CA: true");
                            LOGGER.info("    path length constraint: " + pathLenConstraint);
                            if (pathLenConstraint == 0) {
                                LOGGER.info("    only end entity may follow");
                            } else if (pathLenConstraint == Integer.MAX_VALUE) {
                                LOGGER.info("    unlimited CAs or end entities may follow");
                            } else {
                                LOGGER.info("    " + pathLenConstraint + " CAs may follow");
                            }
                        } else {
                            LOGGER.info("    is CA: false");
                        }
                    }
                });
            }
            if (!(nonCriticalExtensions = certificate.getNonCriticalExtensionOIDs()).isEmpty()) {
                LOGGER.info("Active Non Critical Extensions:");
                nonCriticalExtensions.stream().forEach(item -> {
                    if ("2.16.840.1.113730.1.1".equals(item)) {
                        LOGGER.info("  Netscape Certificate Type (OID: 2.16.840.1.113730.1.1)");
                    } else if ("2.5.29.17".equals(item)) {
                        LOGGER.info("  Subject Alternative Name (OID: 2.5.29.17)");
                        try {
                            Collection<List<?>> subjectAlternativeNames = certificate.getSubjectAlternativeNames();
                            for (List<?> entry : subjectAlternativeNames) {
                                int key = (Integer)entry.get(0);
                                String keyName = "";
                                switch (key) {
                                    case 0: {
                                        keyName = "otherName";
                                        break;
                                    }
                                    case 1: {
                                        keyName = "rfc822Name";
                                        break;
                                    }
                                    case 2: {
                                        keyName = "dNSName";
                                        break;
                                    }
                                    case 3: {
                                        keyName = "x400Address";
                                        break;
                                    }
                                    case 4: {
                                        keyName = "directoryName";
                                        break;
                                    }
                                    case 5: {
                                        keyName = "ediPartyName";
                                        break;
                                    }
                                    case 6: {
                                        keyName = "uniformResourceIdentifier";
                                        break;
                                    }
                                    case 7: {
                                        keyName = "iPAddress";
                                        break;
                                    }
                                    case 8: {
                                        keyName = "registeredID";
                                    }
                                }
                                String value = (String)entry.get(1);
                                LOGGER.info("    " + keyName + " : " + value);
                            }
                        }
                        catch (CertificateParsingException e) {
                            LOGGER.error(e.getMessage(), (Throwable)e);
                        }
                    } else if ("2.5.29.37".equals(item)) {
                        LOGGER.info("  Extended Key Usage (OID: 2.5.29.37)");
                        try {
                            List<String> extendedKeyUsage = certificate.getExtendedKeyUsage();
                            extendedKeyUsage.stream().forEach(extKexUsage -> {
                                if ("1.3.6.1.5.5.7.3.2".equals(extKexUsage)) {
                                    LOGGER.info("    'clientAuth' (OID: 1.3.6.1.5.5.7.3.2)");
                                } else if ("1.3.6.1.5.5.7.3.1".equals(extKexUsage)) {
                                    LOGGER.info("    'serverAuth' (OID: 1.3.6.1.5.5.7.3.1)");
                                }
                            });
                        }
                        catch (CertificateParsingException e) {
                            LOGGER.error(e.getMessage(), (Throwable)e);
                        }
                    } else if ("2.5.29.14".equals(item)) {
                        LOGGER.info("  Subject Key Identifier (OID: 2.5.29.14)");
                    } else if ("2.5.29.35".equals(item)) {
                        LOGGER.info("  Authority Key Identifier (OID: 2.5.29.35)");
                    }
                });
            }
            LOGGER.info("");
        }
    }

    public static String asPEMString(X509Certificate cert) throws IOException, CertificateEncodingException {
        String lineSeparator = "\n";
        Base64.Encoder encoder = Base64.getMimeEncoder(64, lineSeparator.getBytes());
        String encodedCert = new String(encoder.encode(cert.getEncoded()));
        return "-----BEGIN CERTIFICATE-----" + lineSeparator + encodedCert + lineSeparator + "-----END CERTIFICATE-----";
    }

    public static String getDistinguishedName(X509Certificate cert) throws CertificateEncodingException {
        X500Name x500name = new JcaX509CertificateHolder(cert).getSubject();
        List<String> rdns = Arrays.asList(x500name.toString().split(","));
        Collections.reverse(rdns);
        return rdns.stream().collect(Collectors.joining(", "));
    }

    public static String extractDistinguishedNameQualifier(X509Certificate cert) throws CertificateEncodingException {
        X500Name x500name = new JcaX509CertificateHolder(cert).getSubject();
        if (x500name.getRDNs(BCStyle.DN_QUALIFIER)[0].size() > 0) {
            return IETFUtils.valueToString((ASN1Encodable)x500name.getRDNs(BCStyle.DN_QUALIFIER)[0].getFirst().getValue());
        }
        return null;
    }

    public static String extractFirstCommonName(X509Certificate cert) throws CertificateEncodingException {
        X500Name x500name = new JcaX509CertificateHolder(cert).getSubject();
        if (x500name.getRDNs(BCStyle.CN)[0].size() > 0) {
            return IETFUtils.valueToString((ASN1Encodable)x500name.getRDNs(BCStyle.CN)[0].getFirst().getValue());
        }
        return null;
    }

    public static List<String> extractCommonNames(X509Certificate cert) throws CertificateEncodingException {
        X500Name x500name = new JcaX509CertificateHolder(cert).getSubject();
        RDN cn = x500name.getRDNs(BCStyle.CN)[0];
        if (cn.size() > 0) {
            ArrayList<String> cns = new ArrayList<String>();
            AttributeTypeAndValue[] values = cn.getTypesAndValues();
            for (int i = 0; i < values.length; ++i) {
                cns.add(IETFUtils.valueToString((ASN1Encodable)values[i].getValue()));
            }
            return cns;
        }
        return null;
    }

    public static String extractFirstOrganizationUnit(X509Certificate cert) throws CertificateEncodingException {
        X500Name x500name = new JcaX509CertificateHolder(cert).getSubject();
        return IETFUtils.valueToString((ASN1Encodable)x500name.getRDNs(BCStyle.OU)[0].getFirst().getValue());
    }

    public static List<String> extractOrganizationUnits(X509Certificate cert) throws CertificateEncodingException {
        X500Name x500name = new JcaX509CertificateHolder(cert).getSubject();
        RDN ou = x500name.getRDNs(BCStyle.OU)[0];
        if (ou.size() > 0) {
            ArrayList<String> ous = new ArrayList<String>();
            AttributeTypeAndValue[] values = ou.getTypesAndValues();
            for (int i = 0; i < values.length; ++i) {
                ous.add(IETFUtils.valueToString((ASN1Encodable)values[i].getValue()));
            }
            return ous;
        }
        return null;
    }

    public static List<String> extractDomainComponents(X509Certificate cert) throws CertificateEncodingException {
        X500Name x500name = new JcaX509CertificateHolder(cert).getSubject();
        RDN dc = x500name.getRDNs(BCStyle.DC)[0];
        if (dc.size() > 0) {
            ArrayList<String> dcs = new ArrayList<String>();
            AttributeTypeAndValue[] values = dc.getTypesAndValues();
            for (int i = 0; i < values.length; ++i) {
                dcs.add(IETFUtils.valueToString((ASN1Encodable)values[i].getValue()));
            }
            return dcs;
        }
        return null;
    }

    public static String extractOrganization(X509Certificate cert) throws CertificateEncodingException {
        X500Name x500name = new JcaX509CertificateHolder(cert).getSubject();
        if (x500name.getRDNs(BCStyle.O)[0].size() > 0) {
            return IETFUtils.valueToString((ASN1Encodable)x500name.getRDNs(BCStyle.O)[0].getFirst().getValue());
        }
        return null;
    }

    public static String extractCountryCode(X509Certificate cert) throws CertificateEncodingException {
        X500Name x500name = new JcaX509CertificateHolder(cert).getSubject();
        if (x500name.getRDNs(BCStyle.C)[0].size() > 0) {
            return IETFUtils.valueToString((ASN1Encodable)x500name.getRDNs(BCStyle.C)[0].getFirst().getValue());
        }
        return null;
    }

    public static String extractLocality(X509Certificate cert) throws CertificateEncodingException {
        X500Name x500name = new JcaX509CertificateHolder(cert).getSubject();
        if (x500name.getRDNs(BCStyle.L)[0].size() > 0) {
            return IETFUtils.valueToString((ASN1Encodable)x500name.getRDNs(BCStyle.L)[0].getFirst().getValue());
        }
        return null;
    }

    public static String extractState(X509Certificate cert) throws CertificateEncodingException {
        X500Name x500name = new JcaX509CertificateHolder(cert).getSubject();
        if (x500name.getRDNs(BCStyle.ST)[0].size() > 0) {
            return IETFUtils.valueToString((ASN1Encodable)x500name.getRDNs(BCStyle.ST)[0].getFirst().getValue());
        }
        return null;
    }
}

