#!/bin/sh

# ------------------------------------------------------------
# Company:  Software- und Organisations-Service GmbH
# Date:     2025-04-15
# Purpose:  execute YADE Client
# Platform: AIX, Linux, MacOS, Solaris: bash, ksh, zsh, dash
# ------------------------------------------------------------
#
# Example:  ./yade.sh --java-options="-Xmx64m" --settings=./yade.xml --profile=copy_sftp
#           runs the "copy_sftp" profile from the indicated settings file and limits Java Heap Space

set -e

# ------------------------------
# Initialize variables
# ------------------------------

YADE_BIN=$(cd "$(dirname "$0")" >/dev/null && pwd)
YADE_HOME="$(cd "${YADE_BIN}/.." >/dev/null && pwd)"

yade_script="$0"
yade_java_options="${JAVA_OPTIONS}"
log_level=INFO
settings=
settings_arg=
return_values=
return_values_arg=
java_bin=
classpath=
log4j_property=
hostname=$(hostname 2>/dev/null)
verbose=0

if [ -z "$LANG" ]
then
    locale_bin=$(which locale 2>/dev/null || echo "")
    if [ -n "${locale_bin}" ]
    then
        LANG="$(locale -a | grep -i "en_us.utf" | head -n 1)"
        export LANG
    fi
fi

Log_Verbose()
{
    if [ "${verbose}" -gt 0 ]
    then
        >&2 echo ""
        >&2 echo "Verbose: $*"
        >&2 echo ""
    fi
}

Output_Header()
{
    echo " "
    echo "****************************************************************************************"
    echo "CLIENT START : $(TZ=${YADE_TZ} date +'%Y-%m-%d %H:%M:%S%:z %Z')"
    echo "HOSTNAME     : ${hostname}"
    echo "USER         : ${USER}"
    echo "CALL         : ${yade_script} $*"
    echo " "    
}

Output_Footer()
{
    echo " "
    echo "****************************************************************************************"
    echo "CLIENT END   : $(TZ=${YADE_TZ} date +'%Y-%m-%d %H:%M:%S%:z %Z'), exit code $1"
    echo " "
}

# ------------------------------
# Process command line options
# ------------------------------

Arguments()
{
    for option in "$@"
    do
        case "${option}" in
            -h|--help)                          
                                                ;;
            --java-options=*|-java-options=*)   yade_java_options=$(echo "${option}" | sed 's/--java-options=//' | sed 's/-java-options=//' | sed 's/^"//' | sed 's/"$//' )
                                                ;;
            --log-level=*|-log-level=*)         log_level=$(echo "${option}" | sed 's/--log-level=//' | sed 's/-log-level=//')
                                                ;;
            --settings=*|-settings=*)           settings=$(echo "${option}" | sed 's/--settings=//' | sed 's/-settings=//' | sed 's/^"//' | sed 's/"$//' | sed 's/^\(.*\)\/$/\1/')
                                                ;;
            --return-values=*|-return-values=*) return_values=$(echo "${option}" | sed 's/--return-values=//' | sed 's/-return-values=//' | sed 's/^"//' | sed 's/"$//' | sed 's/^\(.*\)\/$/\1/')
                                                ;;
            --verbose|-v)                       verbose=1
                                                ;;
            --*|-*)                             
                                                ;;
            esac
    done

    if [ -z "${settings}" ] && [ -n "${JS7_YADE_CONFIG_DIR}" ] 
    then
        settings_arg="--settings=\"${JS7_YADE_CONFIG_DIR}/yade.xml\""
    fi
    
    if [ -z "${return_values}" ] && [ -n "${JS7_RETURN_VALUES}" ]
    then
        return_values_arg="--return-values=\"${JS7_RETURN_VALUES}\""
    fi

    if [ -n "${JS7_RETURN_VALUES}" ]
    then
        test -z "${JS7_AGENT_TZ}" || YADE_TZ="${JS7_AGENT_TZ}"
        test -n "${YADE_TZ}" || YADE_TZ="Etc/UTC"
    else
        test -n "${YADE_TZ}" || YADE_TZ=$(date +%Z)    
    fi
    
    if [ -n "${log_level}" ]
    then
        YADE_LOG_LEVEL=$(echo "${log_level}" | tr '[:lower:]' '[:upper:]')
    fi
    
    Output_Header "$@"
}

# ------------------------------
# Execute YADE
# ------------------------------

Process()
{
    export YADE_HOME
    export YADE_BIN
    export YADE_TZ
    export YADE_LOG_LEVEL

    # set java class path
    if [ ! -d "${YADE_HOME}"/lib ]
    then
        >&2 echo "ERROR: Directory ${YADE_HOME}/lib not found."
        exit 1
    fi

    classpath="${YADE_HOME}/lib/patches/*:${YADE_HOME}/lib/user_lib/*:${YADE_HOME}/lib/3rd-party/*:${YADE_HOME}/lib/sos/*"
    
    test -z "${log4j_property}" && test -f "${YADE_HOME}"/lib/log4j2.xml && log4j_property="-Dlog4j.configurationFile=file://${YADE_HOME}/lib/log4j2.xml"

    java_bin=$(which java 2>/dev/null || echo "")
    test -n "${JAVA_HOME}" && test -x "${JAVA_HOME}/bin/java" && java_bin="${JAVA_HOME}/bin/java"

    if [ -z "${java_bin}" ]
    then
        >&2 echo "ERROR: Java not found"
        >&2 echo "INFO: set JAVA_HOME environment variable"
        exit 5
    fi

    yade_xmx_java_option=$(echo "${yade_java_options}" | { grep "\-Xmx" || :; })
    test -n "${yade_xmx_java_option}" || yade_java_options="${yade_java_options} -Xmx32m"

    Log_Verbose "${java_bin} ${log4j_property} ${yade_java_options} -classpath ${classpath} com.sos.yade.engine.YADEEngineMain $* ${settings_arg} ${return_values_arg}"
    "${java_bin}" ${log4j_property} ${yade_java_options} -classpath "${classpath}" com.sos.yade.engine.YADEEngineMain $@ ${settings_arg} ${return_values_arg}
}

# ------------------------------
# Cleanup temporary resources
# ------------------------------

End()
{
    Output_Footer "$?"

    unset yade_java_options
    unset yade_xmx_java_option
    unset log_level
    unset settings
    unset settings_arg
    unset return_values
    unset return_values_arg
    unset java_bin
    unset classpath
    unset log4j_property
    unset verbose
    
    set +e
}

# ------------------------------
# Enable trap and start
# ------------------------------

trap 'End EXIT' EXIT
trap 'End SIGTERM' TERM
trap 'End SIGINT' INT

Arguments "$@"
Process "$@"
